package main

import (
	"context"
	"fmt"
	"net/http"
	_ "net/http/pprof"
	"os"
	"os/signal"
	"syscall"

	"github.com/spf13/pflag"
	"go.uber.org/zap"

	"a.yandex-team.ru/library/go/core/log"
	xzap "a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/security/gideon/nuvault/internal/app"
	"a.yandex-team.ru/security/gideon/nuvault/internal/config"
)

var Flags = struct {
	ConfigPath string
}{
	ConfigPath: "/app.yaml",
}

func main() {
	maxprocs.AdjustAuto()
	pflag.StringVarP(&Flags.ConfigPath, "cfg", "c", Flags.ConfigPath, "config path")
	pflag.Parse()

	go func() {
		_ = http.ListenAndServe("localhost:6060", nil)
	}()

	cfg, err := config.NewConfig(Flags.ConfigPath)
	if err != nil {
		fmt.Printf("failed to parse config: %v\n", err)
		os.Exit(1)
	}

	logLevel, err := log.ParseLevel(cfg.LogLevel)
	if err != nil {
		fmt.Printf("failed to parse log level (%s): %v\n", cfg.LogLevel, err)
		os.Exit(1)
	}

	logger, err := xzap.NewDeployLogger(logLevel, zap.AddStacktrace(zap.ErrorLevel))
	if err != nil {
		fmt.Printf("failed to initialize logger: %v\n", err)
		os.Exit(1)
	}

	instance, err := app.NewApp(cfg, logger)
	if err != nil {
		fmt.Printf("failed to initialize application: %v\n", err)
		os.Exit(1)
	}

	errChan := make(chan error, 1)
	okChan := make(chan struct{}, 1)
	go func() {
		err := instance.Start()
		if err != nil {
			errChan <- err
			logger.Error("start fail", log.Error(err))
		} else {
			okChan <- struct{}{}
		}
	}()

	stopChan := make(chan os.Signal, 1)
	signal.Notify(stopChan, syscall.SIGINT, syscall.SIGTERM)
	select {
	case <-stopChan:
		logger.Info("shutting down gracefully by signal")

		ctx, cancel := context.WithTimeout(context.Background(), config.ShutdownDeadline)
		defer cancel()

		if err := instance.Shutdown(ctx); err != nil {
			logger.Error("failed to shutdown beaver", log.Error(err))
		}
	case <-okChan:
		return
	case <-errChan:
		return
	}
}
