package yaver

import (
	"context"
	"errors"
	"fmt"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/yav"
	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
	"a.yandex-team.ru/security/gideon/nuvault/pkg/nuvrpc"
)

const (
	fetchTimeout = 10 * time.Second
)

type Yaver struct {
	cfg      Config
	yavc     yav.Client
	ctx      context.Context
	cancelFn context.CancelFunc
	log      log.Logger
	lastSync time.Time
	mu       sync.RWMutex
	secVers  map[string]*nuvrpc.Secret
}

func NewYaver(cfg Config, l log.Logger) (*Yaver, error) {
	if cfg.AuthToken == "" {
		return nil, errors.New("empty auth token")
	}

	if len(cfg.Secrets) == 0 {
		return nil, errors.New("no secrets configured")
	}

	logger := l.WithName("yaver")
	yavc, err := httpyav.NewClient(
		httpyav.WithOAuthToken(cfg.AuthToken),
		httpyav.WithLogger(logger),
	)
	if err != nil {
		return nil, fmt.Errorf("failed to create yav client: %w", err)
	}

	ctx, cancelCtx := context.WithCancel(context.Background())
	return &Yaver{
		cfg:      cfg,
		yavc:     yavc,
		ctx:      ctx,
		cancelFn: cancelCtx,
		log:      logger,
		secVers:  make(map[string]*nuvrpc.Secret, len(cfg.Secrets)),
	}, nil
}

func (y *Yaver) Sync(stopOnError bool) error {
	c := 0
	secrets := make(map[string]yav.Version)
	fetch := func(secretUUID string) error {
		ctx, cancel := context.WithTimeout(y.ctx, fetchTimeout)
		defer cancel()

		rsp, err := y.yavc.GetVersion(ctx, secretUUID)
		if err != nil {
			return fmt.Errorf("failed to get secret (%s): %w", secretUUID, err)
		}

		if rsp.Status != yav.StatusOK {
			y.log.Error(
				"not OK status for secret",
				log.String("secret", secretUUID),
				log.String("status", rsp.Status),
				log.String("code", rsp.Code),
			)
			return nil
		}

		secrets[secretUUID] = rsp.Version
		c++
		return nil
	}

	for _, secretUUID := range y.cfg.Secrets {
		err := fetch(secretUUID)
		if err != nil {
			if stopOnError {
				return err
			}

			y.log.Error("failed to sync secrets", log.Error(err))
			continue
		}
	}

	y.mu.Lock()
	for k, v := range secrets {
		y.secVers[k] = yavToSecret(v)
	}
	y.lastSync = time.Now()
	y.mu.Unlock()

	y.log.Info("secrets synced", log.Int("count", c))
	return nil
}

func (y *Yaver) GetSecret(secretUUID string) (*nuvrpc.Secret, error) {
	y.mu.RLock()
	defer y.mu.RUnlock()

	ver, ok := y.secVers[secretUUID]
	if !ok {
		return nil, fmt.Errorf("secret %s not found or not allowed", secretUUID)
	}

	return ver, nil
}

func (y *Yaver) Start() {
	var err error
	for {
		toNextSync := time.Until(
			time.Now().Add(y.cfg.SyncPeriod).Truncate(y.cfg.SyncPeriod),
		)

		select {
		case <-y.ctx.Done():
			return
		case <-time.After(toNextSync):
			err = y.Sync(false)
			if err != nil {
				y.log.Warn("sync fail", log.Error(err))
			}
		}
	}
}

func (y *Yaver) InSync() error {
	y.mu.RLock()
	defer y.mu.RUnlock()

	sinceSync := time.Since(y.lastSync)
	if sinceSync > y.cfg.SyncDrift {
		return fmt.Errorf("secrets out of sync: %s (cur drift) > %s (allowed)", sinceSync, y.cfg.SyncDrift)
	}

	return nil
}

func (y *Yaver) Shutdown() {
	y.cancelFn()
}

func yavToSecret(yavVer yav.Version) *nuvrpc.Secret {
	out := &nuvrpc.Secret{
		SecretUuid:  yavVer.SecretUUID,
		VersionUuid: yavVer.VersionUUID,
		Values:      make(map[string]string, len(yavVer.Values)),
	}

	for _, val := range yavVer.Values {
		// TODO(buglloc): base64 support?
		out.Values[val.Key] = val.Value
	}
	return out
}
