package nuvault

import (
	"context"
	"fmt"
	"time"

	"google.golang.org/grpc"
	"google.golang.org/protobuf/types/known/emptypb"

	"a.yandex-team.ru/infra/yp_service_discovery/golang/resolver"
	"a.yandex-team.ru/infra/yp_service_discovery/golang/wrapper/grpcresolver"
	"a.yandex-team.ru/security/gideon/nuvault/pkg/nuvrpc"
)

const (
	DefaultAddr            = "nuvault.Api"
	DefaultRetries         = 3
	DefaultTimeout         = 500 * time.Millisecond
	DefaultBackoffDuration = 100 * time.Millisecond
	ServerName             = "nuvault.sec.yandex.net"
	serviceConfigJSON      = `{"loadBalancingPolicy":"round_robin"}`
)

var (
	DefaultClusters = []string{
		resolver.ClusterSAS,
		resolver.ClusterVLA,
		resolver.ClusterIVA,
	}
)

type Client struct {
	conn          *grpc.ClientConn
	nuvService    nuvrpc.NuVaultServiceClient
	commonService nuvrpc.CommonServiceClient
}

func NewClient(opts ...Option) (*Client, error) {
	cfg := config{
		addr:            DefaultAddr,
		timeout:         DefaultTimeout,
		backoffDuration: DefaultBackoffDuration,
		maxRetries:      DefaultRetries,
		crtRequired:     true,
	}

	for _, opt := range opts {
		opt(&cfg)
	}

	addr := cfg.addr
	if addr == DefaultAddr {
		addr = grpcresolver.GrpcTarget(DefaultAddr, DefaultClusters...)
	}
	dialOpts, err := cfg.buildDialOpts()
	if err != nil {
		return nil, fmt.Errorf("failed to build gRPC opts: %w", err)
	}
	conn, err := grpc.Dial(addr, dialOpts...)
	if err != nil {
		return nil, fmt.Errorf("failed to create gRPC connection: %w", err)
	}

	return &Client{
		conn:          conn,
		nuvService:    nuvrpc.NewNuVaultServiceClient(conn),
		commonService: nuvrpc.NewCommonServiceClient(conn),
	}, nil
}

func (nv *Client) Close() error {
	return nv.conn.Close()
}

func (nv *Client) GetSecret(ctx context.Context, key string) (*nuvrpc.Secret, error) {
	rsp, err := nv.nuvService.GetSecret(ctx, &nuvrpc.GetSecretRequest{
		SecretUuid: key,
	})
	if err != nil {
		return nil, err
	}

	return rsp.Secret, nil
}

func (nv *Client) PingReadiness(ctx context.Context) error {
	_, err := nv.commonService.Readiness(ctx, &emptypb.Empty{})
	return err
}

func (nv *Client) PingLiveness(ctx context.Context) error {
	_, err := nv.commonService.Liveness(ctx, &emptypb.Empty{})
	return err
}
