package nuvault

import (
	"time"

	grpcRetry "github.com/grpc-ecosystem/go-grpc-middleware/retry"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	grpcCredentials "google.golang.org/grpc/credentials"
)

type Option func(*config)

type config struct {
	addr            string
	maxRetries      int
	timeout         time.Duration
	backoffDuration time.Duration
	crtRequired     bool
	clientCert      string
}

// WithNoAuth disables client authentication
func WithNoAuth() Option {
	return func(cfg *config) {
		cfg.crtRequired = false
	}
}

// WithMutualAuth sets path to TLS certificate to use for mutual authentication
func WithMutualAuth(certPath string) Option {
	return func(cfg *config) {
		cfg.crtRequired = true
		cfg.clientCert = certPath
	}
}

// WithAddr overrides gRPC API address
// Default: client resolves "nuvault.Api" endpoint and connect to it
func WithAddr(addr string) Option {
	return func(cfg *config) {
		cfg.addr = addr
	}
}

// WithMaxRetries sets the maximum number of retries
// Default: 3
func WithMaxRetries(retries int) Option {
	return func(cfg *config) {
		cfg.maxRetries = retries
	}
}

// WithBackoffDuration sets base exponential backoff timeout.
// Default: 100ms
func WithBackoffDuration(duration time.Duration) Option {
	return func(cfg *config) {
		cfg.backoffDuration = duration
	}
}

// WithTimeout sets the gRPC per call timeout
// Default: 500ms
func WithTimeout(timeout time.Duration) Option {
	return func(cfg *config) {
		cfg.timeout = timeout
	}
}

func (c *config) buildDialOpts() ([]grpc.DialOption, error) {
	if c.clientCert == "" && c.crtRequired {
		return nil, errNoClientCert
	}

	tlsConfig, err := newTLSConfig(c.clientCert)
	if err != nil {
		return nil, err
	}

	grpcRetry.WithBackoff(grpcRetry.BackoffLinear(100 * time.Millisecond))
	return []grpc.DialOption{
		grpc.WithDefaultServiceConfig(serviceConfigJSON),
		grpc.WithDisableServiceConfig(),
		grpc.WithTransportCredentials(
			grpcCredentials.NewTLS(tlsConfig),
		),
		grpc.WithUnaryInterceptor(
			grpcRetry.UnaryClientInterceptor(
				grpcRetry.WithMax(uint(c.maxRetries)),
				grpcRetry.WithPerRetryTimeout(c.timeout),
				grpcRetry.WithCodes(codes.Aborted, codes.Internal, codes.DataLoss),
				grpcRetry.WithBackoff(grpcRetry.BackoffExponentialWithJitter(c.backoffDuration, 0.2)),
			),
		),
	}, nil
}
