package nuvault

import (
	"crypto/tls"
	"crypto/x509"
	"errors"
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/certifi"
)

var (
	errNoClientCert  = errors.New("no client certificate")
	errNoPrivKeyCert = errors.New("no private key in the client certificate")
	errNoPubKeyCert  = errors.New("no public key in the client certificate")
)

func newTLSConfig(certPath string) (*tls.Config, error) {
	rootCAs, err := certifi.NewCertPoolSystem()
	if err != nil {
		return nil, fmt.Errorf("failed to create cert pool: %s", err)
	}

	tlsConfig := &tls.Config{
		ServerName: ServerName,
		RootCAs:    rootCAs,
	}

	if certPath != "" {
		loadCert := func() ([]tls.Certificate, error) {
			clientCert, err := tls.LoadX509KeyPair(certPath, certPath)
			if err != nil {
				return nil, err
			}

			if clientCert.PrivateKey == nil {
				return nil, errNoPrivKeyCert
			}

			if len(clientCert.Certificate) == 0 {
				return nil, errNoPubKeyCert
			}

			pubCert, err := x509.ParseCertificate(clientCert.Certificate[0])
			if err != nil {
				return nil, err
			}

			now := time.Now()
			if now.Before(pubCert.NotBefore) {
				return nil, fmt.Errorf("certificate can't be used before %s (now: %s)", pubCert.NotBefore, now)
			}

			if now.After(pubCert.NotAfter) {
				return nil, fmt.Errorf("certificate can't be used after %s (now: %s)", pubCert.NotAfter, now)
			}

			return []tls.Certificate{clientCert}, nil
		}

		tlsConfig.Certificates, err = loadCert()
		if err != nil {
			return nil, fmt.Errorf("failed to cload client crt (%s): %w", certPath, err)
		}
	}

	return tlsConfig, nil
}
