#include "gideon.h"

#include <curl/curl.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/writer/json.h>

#include <util/generic/scope.h>
#include <util/string/cast.h>
#include <util/stream/str.h>
#include <util/stream/file.h>
#include <sys/utsname.h>

namespace {
    const long kConnectionTimeout = 100;

    size_t onCurlWrite(void *buf, size_t size, size_t nmemb, void *userp) {
        if (!userp) {
            return 0;
        }

        TStringStream &os = *static_cast<TStringStream *>(userp);
        size_t len = size * nmemb;
        os.Write(static_cast<char *>(buf), len);
        return len;
    }

    bool httpPost(const TString &url, const TString &data, TStringStream *response) {
        if (response == nullptr) {
            return false;
        }

        curl_global_init(CURL_GLOBAL_ALL & ~CURL_GLOBAL_SSL);
        CURL *curl = curl_easy_init();
        if (!curl) {
            curl_global_cleanup();
            return false;
        }

        Y_DEFER {
            curl_easy_cleanup(curl);
            curl_global_cleanup();
        };

        struct curl_slist *headers = nullptr;
        headers = curl_slist_append(headers, "Accept: application/json");
        headers = curl_slist_append(headers, "Content-Type: application/json");
        if (headers == nullptr) {
            return false;
        }
        Y_DEFER {
            curl_slist_free_all(headers);
        };

        curl_easy_setopt(curl, CURLOPT_ABSTRACT_UNIX_SOCKET, "gideon");
        curl_easy_setopt(curl, CURLOPT_URL, url.data());
        curl_easy_setopt(curl, CURLOPT_HTTPHEADER, headers);
        curl_easy_setopt(curl, CURLOPT_USERAGENT, "pam_gideon");
        curl_easy_setopt(curl, CURLOPT_TIMEOUT_MS, kConnectionTimeout);
        curl_easy_setopt(curl, CURLOPT_POSTFIELDS, data.data());
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, onCurlWrite);
        curl_easy_setopt(curl, CURLOPT_WRITEDATA, response);

        if (curl_easy_perform(curl) != CURLE_OK) {
            return false;
        }

        long httpCode = 0;
        curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &httpCode);
        return httpCode == 200;
    }
}


namespace NGideon {

    const TString &KernelVersion() {
        static TString version;
        if (!version) {
            struct utsname name{};
            Y_VERIFY(!uname(&name));
            version = name.release;
        }
        return version;
    }

    int CompareKernelVersion(const TStringBuf &version) {
        return strverscmp(KernelVersion().c_str(), version.data());
    }

    bool CheckPrerequisites() {
        return CompareKernelVersion("4.19") >= 0;
    }

    ui32 CurrentGideonID() {
        TFileInput input("/proc/self/sessionid");
        return FromString<ui32>(input.ReadAll());
    }

    TString StartSession(const TSessionInfo *sessionInfo) {
        NJsonWriter::TBuf body(NJsonWriter::HEM_UNSAFE);
        body.BeginObject();
        body.WriteKey("user");
        body.WriteString(sessionInfo->UserName);
        body.WriteKey("tty");
        body.WriteString(sessionInfo->TTYName);
        body.EndObject();

        TStringStream respBody;
        respBody.Reserve(512);
        bool ok = httpPost("http://localhost/tracer/session", body.Str(), &respBody);
        if (!ok) {
            ythrow TSystemError() << "can't start Gideon session";
        }

        NJson::TJsonValue rsp;
        NJson::ReadJsonTree(respBody.Str(), &rsp, true);
        return rsp["external_session_id"].GetString();
    }
}
