#include "pam_utils.h"

#include <util/generic/strbuf.h>
#include <util/generic/yexception.h>
#include <security/pam_modules.h>
#include <security/pam_ext.h>
#include <syslog.h>

namespace NGideon::NPam {

    TStringBuf ServiceName(pam_handle_t *pamh) {
        const char *serviceName = nullptr;
        int res = pam_get_item(pamh, PAM_SERVICE, reinterpret_cast<const void **>(&serviceName));
        if (res != PAM_SUCCESS || !serviceName || !*serviceName) {
            return TStringBuf{};
        }

        return serviceName;
    }

    TStringBuf UserName(pam_handle_t *pamh) {
        const char *user_name = nullptr;
        int res = pam_get_user(pamh, &user_name, nullptr);
        if (res != PAM_SUCCESS) {
            ythrow TSystemError(res) << "pam_get_user fail";
        }

        return TStringBuf(user_name);
    }

    TStringBuf RawAuthInfo(pam_handle_t *pamh) {
        const char *ssh_user_auth = pam_getenv(pamh, "SSH_AUTH_INFO_0");
        if (!ssh_user_auth) {
            return TStringBuf();
        }

        return TStringBuf(ssh_user_auth);
    }

    TStringBuf TTYName(pam_handle_t *pamh) {
        const char *tty = nullptr;
        int res = pam_get_item(pamh, PAM_TTY, reinterpret_cast<const void **>(&tty));
        if (res == PAM_SUCCESS && tty && *tty) {
            return tty;
        }

        tty = ttyname(STDIN_FILENO);
        if (!tty || !*tty) {
            return TStringBuf{};
        }

        return tty;
    }

    NSsh::TAuthInfo AuthInfo(pam_handle_t *pamh) {
        const char *ssh_user_auth = pam_getenv(pamh, "SSH_AUTH_INFO_0");
        if (!ssh_user_auth) {
            return NGideon::NSsh::TAuthInfo{};
        }

        try {
            return NSsh::ParseAuthInfo(ssh_user_auth);
        } catch (const yexception &e) {
            pam_syslog(pamh, LOG_ERR, "failed to parse SSH_AUTH_INFO_0 '%s': %s", ssh_user_auth, e.AsStrBuf().data());
        }

        return NSsh::TAuthInfo{};
    }
}