#include "ssh_utils.h"

#include <contrib/libs/openssl/include/openssl/sha.h>
#include <library/cpp/string_utils/base64/base64.h>
#include <util/generic/strbuf.h>
#include <util/string/builder.h>

namespace NGideon::NSsh {
    namespace {
        inline EAuthMethod ParseAuthMethod(TStringBuf methodName) {
            if (methodName == "publickey") {
                return EAuthMethod::PublicKey;
            }

            if (methodName == "password") {
                return EAuthMethod::Password;
            }

            ythrow TParseError() << "unexpected auth method: " << methodName;
        }

        inline TString SshSha256(const char *data, size_t len) {
            TString dStr(SHA256_DIGEST_LENGTH, '\x00');
            SHA256(reinterpret_cast<const unsigned char *>(data), len, reinterpret_cast<unsigned char *>(dStr.begin()));

            const auto &base64 = Base64Encode(dStr);
            // Remove padding
            return base64.substr(0, base64.size() - 1);
        }

        TString SshFingerprint(TStringBuf encodedKey) {
            const TString &key = Base64DecodeUneven(encodedKey);
            return "SHA256:" + SshSha256(key.data(), key.size());
        }
    }

    TString TAuthInfo::Description() const {
        TStringBuilder out;
        out << "auth_method=" << AuthMethodName(this->AuthMethod);
        if (!this->KeyTypeName.empty()) {
            out << " key_algorithm=" << this->KeyTypeName;
        }

        if (!this->Fingerprint.empty()) {
            out << " fingerprint=" << this->Fingerprint;
        }

        return out;
    }

    TAuthInfo ParseAuthInfo(TStringBuf source) {
        source = source.Before('\n');
        TStringBuf tok = source.NextTok(' ');
        if (tok.empty()) {
            ythrow TParseError() << "auth method expected";
        }

        switch (ParseAuthMethod(tok)) {
            case EAuthMethod::Password:
                return TAuthInfo{.AuthMethod = EAuthMethod::Password};
            case EAuthMethod::PublicKey: {
                tok = source.NextTok(' ');
                if (tok.empty()) {
                    ythrow TParseError() << "key type expected";
                }

                const TString keyTypeName(tok);

                tok = source.NextTok(' ');
                if (tok.empty()) {
                    ythrow TParseError() << "pubkey expected";
                }

                return TAuthInfo{
                        .AuthMethod = EAuthMethod::PublicKey,
                        .KeyTypeName = keyTypeName,
                        .Fingerprint = SshFingerprint(tok),
                };
            }
            default:
                return TAuthInfo{};
        }
    }
}
