#include <util/generic/yexception.h>
#include <security/pam_appl.h>
#include <security/pam_modules.h>
#include <security/pam_ext.h>
#include <syslog.h>

#include "security/gideon/pam_gideon/lib/seccomp.h"
#include "security/gideon/pam_gideon/lib/gideon.h"
#include "security/gideon/pam_gideon/lib/pam_utils.h"

namespace {
    constexpr TStringBuf kSSHService = "sshd";

    bool suitableService(pam_handle_t *pamh) {
        auto service = NGideon::NPam::ServiceName(pamh);
        return service == kSSHService;
    }

    bool startSession(pam_handle_t *pamh, NGideon::TSessionInfo *sessInfo) {
        sessInfo->UserName = NGideon::NPam::UserName(pamh);
        if (sessInfo->UserName == "root") {
            // temporary breaking glass system
            return false;
        }

        sessInfo->AuthInfo = NGideon::NPam::AuthInfo(pamh);
        if (!NGideon::CheckPrerequisites()) {
            return false;
        }

        sessInfo->GideonID = NGideon::CurrentGideonID();
        if (!sessInfo->GideonID) {
            ythrow yexception() << "no GideonID";
        }

        sessInfo->TTYName = NGideon::NPam::TTYName(pamh);
        sessInfo->SessionID = NGideon::StartSession(sessInfo);
        if (!NGideon::NSeccomp::SetupSeccomp()) {
            ythrow yexception() << "can't setup seccomp profile";
        }

        return true;
    }

}  // anonymous namespace

extern "C" {

PAM_EXTERN int pam_sm_open_session(pam_handle_t *pamh, int flags, int argc, const char **argv) {
    Y_UNUSED(flags);
    Y_UNUSED(argc);
    Y_UNUSED(argv);

    if (!suitableService(pamh)) {
        return PAM_SUCCESS;
    }

    NGideon::TSessionInfo sessInfo;
    bool sessionStarted = false;
    int ret = PAM_ABORT;

    try {
        sessionStarted = startSession(pamh, &sessInfo);
        ret = PAM_SUCCESS;
    } catch (const yexception &e) {
        pam_syslog(pamh, LOG_ERR, "can't handle session: %s", e.AsStrBuf().data());
        ret = PAM_SERVICE_ERR;
    } catch (...) {
        pam_syslog(pamh, LOG_ERR, "can't handle session: shit happens");
        ret = PAM_SERVICE_ERR;
    }

    auto authDesc = sessInfo.AuthInfo.Description();
    if (sessionStarted) {
        pam_syslog(pamh, LOG_INFO,
                   "session started: ret=%d session=%s gideon_session=%u user=%s %s",
                   ret,
                   sessInfo.SessionID.data(),
                   sessInfo.GideonID,
                   sessInfo.UserName.data(),
                   authDesc.data()
        );
    } else {
        pam_syslog(pamh, LOG_INFO,
                   "session %s: ret=%d user=%s %s",
                   ret == PAM_SUCCESS ? "bypassed" : "failed",
                   ret,
                   sessInfo.UserName.data(),
                   authDesc.data()
        );
    }

    return ret;
}

PAM_EXTERN int pam_sm_close_session(pam_handle_t *pamh, int flags, int argc, const char **argv) {
    Y_UNUSED(pamh);
    Y_UNUSED(flags);
    Y_UNUSED(argc);
    Y_UNUSED(argv);
    return PAM_SUCCESS;
}

}
