import os
import signal
import time
import unittest
import yatest
import pamela
import requests_unixsocket

SERVICE = "sshd"
PAM_CONFIG = """
session    optional     %s
"""


def read_auth_log():
    with open("/var/log/auth.log", "rt") as f:
        return f.read()


def read_sessionid():
    with open("/proc/self/sessionid", "rt") as f:
        return int(f.read().strip())


def prepare_pam_module():
    with open("/etc/pam.d/%s" % SERVICE, "a") as f:
        f.write(PAM_CONFIG % yatest.common.binary_path("security/gideon/pam_gideon/pam_gideon.so"))


class TestE2e(unittest.TestCase):
    def setUp(self):
        self.gideon = requests_unixsocket.Session()
        prepare_pam_module()
        self.start_fake_gideon()

    def tearDown(self):
        print("auth log")
        print(read_auth_log())
        if self.gideon_pid != 0:
            os.kill(self.gideon_pid, signal.SIGINT)

    def start_fake_gideon(self):
        self.gideon_pid = os.fork()
        if self.gideon_pid != 0:
            tries = 10
            while tries:
                try:
                    rsp = self.gideon.get("http+unix://\0gideon/ping")
                    if rsp.text == "pong":
                        return
                except:
                    pass

                time.sleep(2)
                tries -= 1
            raise Exception("Fake Gideon not started")

        args = [
            yatest.common.binary_path("security/gideon/pam_gideon/ut/fake-gideon/fake-gideon")
        ]
        os.execv(args[0], args)
        os._exit(os.EX_OK)

    def pump_gideon_sessions(self):
        return self.gideon.get("http+unix://\0gideon/pump").json()

    def test_root(self):
        pid = os.fork()
        with open("/etc/pam.d/%s" % SERVICE, "rt") as f:
            print(f.read())
        if pid == 0:
            pamela.open_session(username="root", service=SERVICE)
            os._exit(os.EX_OK)
            return
        os.waitpid(pid, 0)

    def test_user(self):
        self.pump_gideon_sessions()
        pid = os.fork()
        if pid == 0:
            pamh = pamela.pam_start(username="games", service=SERVICE)
            pamh.put_env("SSH_AUTH_INFO_0", "publickey ecdsa-sha2-nistp256 "
                                            "AAAAE2VjZHNhLXNoYTItbmlzdHAyNTYAAAAIbmlzdHAyNTYAAABBBNBDSGHXl/8R"
                                            "/FloaVY5e4w50n6zqkRdlm3aacXRRk5poJGsJqxc1iBcNzCUBsKMPX5FiKBpV7i"
                                            "+iOvGGFlt018=")
            pamh.open_session()
            pamh.close_session()
            pamela.pam_end(pamh)

            sessions = self.pump_gideon_sessions()
            gideon_session = None
            for sess in sessions:
                if sess["user"] != "games":
                    continue
                gideon_session = sess

            self.assertTrue(gideon_session)
            self.assertEqual(gideon_session["user"], "games")
            external_session_id = gideon_session["external_session_id"]
            self.assertGreater(len(external_session_id), 0)

            log_info = \
                "session started: " + \
                "ret=0 " + \
                ("session=%s " % external_session_id) + \
                ("gideon_session=%d " % read_sessionid()) + \
                "user=games " + \
                "auth_method=publickey " + \
                "key_algorithm=ecdsa-sha2-nistp256 " + \
                "fingerprint=SHA256:9QmY2M1VK5llyKo8E2L72hwTIF9AhAwL4evqf+/DFEQ"

            log_found = False
            with open("/var/log/auth.log", "rt") as f:
                for line in f:
                    print("l: %s" % line)
                    if "pam_gideon" not in line:
                        continue

                    if "session started" not in line:
                        continue

                    if log_info not in line:
                        continue
                    log_found = True

            self.assertTrue(log_found, "entry %r not found in auth-log" % log_info)
            os._exit(os.EX_OK)

        os.waitpid(pid, 0)
