package main

import (
	"context"
	"encoding/json"
	"fmt"
	"net"
	"net/http"
	"sync"

	"github.com/go-chi/chi/v5"
	"github.com/gofrs/uuid"

	"a.yandex-team.ru/library/go/core/log"
)

type emitNewSessionReq struct {
	User string `json:"user"`
}

type emitNewSessionRsp struct {
	ExternalSessionID string `json:"external_session_id"`
}

type session struct {
	User              string `json:"user"`
	ExternalSessionID string `json:"external_session_id"`
}

type SessionStorage struct {
	sessions []session
	mu       sync.RWMutex
}

func (s *SessionStorage) Put(sess session) {
	s.mu.RLock()
	defer s.mu.RUnlock()

	s.sessions = append(s.sessions, sess)
}

func (s *SessionStorage) Pump() []session {
	s.mu.Lock()
	defer s.mu.Unlock()

	sessions := s.sessions[:]
	s.sessions = s.sessions[:0]
	return sessions
}

func newServer(l log.Logger) (func(), error) {
	uuidGen := uuid.NewGen()
	var ss SessionStorage

	r := chi.NewRouter()
	r.Get("/ping", func(w http.ResponseWriter, r *http.Request) {
		_, _ = w.Write([]byte("pong"))
	})

	r.Get("/pump", func(w http.ResponseWriter, r *http.Request) {
		sessions := ss.Pump()
		w.Header().Set("Content-Type", "application/json")
		_ = json.NewEncoder(w).Encode(sessions)
	})

	r.Post("/tracer/session", func(w http.ResponseWriter, r *http.Request) {
		var body emitNewSessionReq
		if err := json.NewDecoder(r.Body).Decode(&body); err != nil {
			l.Error("failed to decode body", log.Error(err))
			http.Error(w, err.Error(), http.StatusBadRequest)
			return
		}

		extSessionUUID, err := uuidGen.NewV4()
		if err != nil {
			l.Error("can't generate sessionid", log.Error(err))
			http.Error(w, fmt.Sprintf("can't generate sessionid: %v", err), http.StatusInternalServerError)
		}
		extSessionID := extSessionUUID.String()

		ss.Put(session{
			User:              body.User,
			ExternalSessionID: extSessionID,
		})

		w.Header().Set("Content-Type", "application/json")
		_ = json.NewEncoder(w).Encode(emitNewSessionRsp{
			ExternalSessionID: extSessionID,
		})
	})

	srv := http.Server{
		Handler: r,
	}

	listener, err := net.Listen("unix", "@gideon")
	if err != nil {
		return nil, err
	}

	go func() {
		err := srv.Serve(listener)
		if err != nil {
			l.Error("server fail", log.Error(err))
		}
	}()

	return func() {
		_ = srv.Shutdown(context.Background())
		_ = listener.Close()
	}, nil
}
