#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <stddef.h>
#include <sys/prctl.h>
#include <linux/seccomp.h>
#include <linux/audit.h>
#include <linux/bpf.h>
#include <linux/filter.h>
#include <linux/unistd.h>


static int install_seccomp(int arch, int error, int sessid) {
    #define BPF_LOG(syscall) \
        BPF_JUMP(BPF_JMP + BPF_JEQ + BPF_K, __NR_##syscall, 0, 1), \
        BPF_STMT(BPF_RET + BPF_K, SECCOMP_RET_LOG | (sessid & SECCOMP_RET_DATA))

    struct sock_filter filter[] = {
        /* validate arch */
        BPF_STMT(BPF_LD + BPF_W + BPF_ABS, (offsetof(struct seccomp_data, arch))),
        BPF_JUMP(BPF_JMP + BPF_JEQ + BPF_K, arch, 1, 0),
        BPF_STMT(BPF_RET + BPF_K, SECCOMP_RET_ERRNO | (error & SECCOMP_RET_DATA)),

        /* load syscall */
        BPF_STMT(BPF_LD + BPF_W + BPF_ABS, (offsetof(struct seccomp_data, nr))),

        /* list of logged syscalls */
        BPF_LOG(execve),
        BPF_LOG(execveat),
        BPF_LOG(ptrace),
        BPF_LOG(connect),

        /* that's fine, allow any unknown syscall to execute  */
        BPF_STMT(BPF_RET + BPF_K, SECCOMP_RET_ALLOW),
    };

    struct sock_fprog prog = {
      .len = (unsigned short)(sizeof(filter) / sizeof(filter[0])),
      .filter = filter,
    };

    if (prctl(PR_SET_NO_NEW_PRIVS, 1, 0, 0, 0)) {
        perror("prctl(NO_NEW_PRIVS)");
        return 1;
    }

    if (prctl(PR_SET_SECCOMP, SECCOMP_MODE_FILTER, &prog)) {
        perror("prctl(PR_SET_SECCOMP)");
        return 1;
    }

    return 0;
}

int main(int argc, char *argv[]) {
    long sessid;
    int opt;
    while ((opt = getopt(argc, argv, "s:")) != -1) {
        switch(opt) {
            case 's':
                sessid = strtol(optarg, NULL, 10);
                if (sessid < 0 || sessid >= 65535) {
                    fprintf(stderr, "invalid sessid, must be 0 < sessid < 65535, current: %ld\n", sessid);
                }
                break;
            case '?':
                fprintf(stderr, "unknown option: %c\n", optopt);
                break;
        }
    }

    if (optind >= argc) {
        fprintf(stderr, "no child configured, usage: %s -- /some/child --child-arg\n", argv[0]);
        return EXIT_FAILURE;
    }


    if (install_seccomp(AUDIT_ARCH_X86_64, EPERM, (int)sessid)) {
        return EXIT_FAILURE;
    }

    execvp(argv[optind], argv+optind);
    perror("execvp()");
    return EXIT_FAILURE;
}
