package main

import (
	"context"
	"flag"
	"fmt"
	"math/rand"
	"os"
	"strconv"
	"time"

	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/security/gideon/speedy-beaver/internal/config"
	"a.yandex-team.ru/security/gideon/speedy-beaver/internal/db"
)

var flags = struct {
	ConfigPath string
	DryMode    bool
	OlderThan  int
}{
	ConfigPath: "/app.yaml",
	OlderThan:  3,
}

func errExit(format string, args ...interface{}) {
	fmt.Printf(format+"\n", args...)
	os.Exit(1)
}

func main() {
	rand.Seed(time.Now().UnixNano())
	maxprocs.AdjustAuto()
	flag.StringVar(&flags.ConfigPath, "cfg", flags.ConfigPath, "config path")
	flag.IntVar(&flags.OlderThan, "older", flags.OlderThan, "cleanup parts older than (in days)")
	flag.BoolVar(&flags.DryMode, "dry", false, "dry run")
	flag.Parse()

	cfg, err := config.NewConfig(flags.ConfigPath)
	if err != nil {
		errExit("failed to parse config: %v", err)
	}

	systemDB, err := db.NewSystemDB(cfg.ClickHouse)
	if err != nil {
		errExit("failed to create system DB: %v", err)
	}

	parts, err := systemDB.ListPartitions(context.Background())
	if err != nil {
		errExit("failed list gideon partitions: %v", err)
	}

	maxPart, err := strconv.Atoi(
		time.Now().Add(-time.Duration(flags.OlderThan*86400) * time.Second).Format("20060102"),
	)
	if err != nil {
		errExit("invalid older range: %v", err)
	}

	var toDrop []int
	fmt.Println("existing partitions:")
	for _, p := range parts {
		fmt.Printf("  - %d", p)
		if p <= maxPart {
			toDrop = append(toDrop, p)
			fmt.Print(" (drop)")
		}
		fmt.Print("\n")
	}
	fmt.Print("\n")

	if flags.DryMode {
		fmt.Println("Dry run mode, exit")
		return
	}

	for _, p := range toDrop {
		err = systemDB.DropPartition(context.Background(), p)
		if err != nil {
			errExit(err.Error())
		}

		fmt.Printf("partition '%d' dropped\n", p)
	}

	fmt.Printf("%d partitions dropped\n", len(toDrop))
}
