package main

import (
	"context"
	"flag"
	"fmt"
	"math/rand"
	"os"
	"os/signal"
	"syscall"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/security/gideon/speedy-beaver/internal/app"
	"a.yandex-team.ru/security/gideon/speedy-beaver/internal/config"
)

var beaverFlags = struct {
	ConfigPath string
}{
	ConfigPath: "/app.yaml",
}

func patchLbDC(cfg *config.Config) {
	dc := os.Getenv("DEPLOY_NODE_DC")
	cfg.Logbroker.Endpoints = []string{dc + ".logbroker.yandex.net"}
	cfg.Lbmask.Endpoint = dc + ".logbroker.yandex.net"
}

func main() {
	rand.Seed(time.Now().UnixNano())
	maxprocs.AdjustYP()
	flag.StringVar(&beaverFlags.ConfigPath, "cfg", beaverFlags.ConfigPath, "config path")
	flag.Parse()

	cfg, err := config.NewConfig(beaverFlags.ConfigPath)
	if err != nil {
		fmt.Printf("failed to parse config: %v\n", err)
		os.Exit(1)
	}

	logLevel, _ := log.ParseLevel(cfg.Logger.Level)
	logger, err := zap.NewDeployLogger(logLevel)
	if err != nil {
		fmt.Printf("failed to initialize logger: %v\n", err)
		os.Exit(1)
	}

	patchLbDC(cfg)
	instance, err := app.NewApp(*cfg, logger)
	if err != nil {
		fmt.Printf("failed to initialize application: %v\n", err)
		os.Exit(1)
	}

	errChan := make(chan error, 1)
	okChan := make(chan struct{}, 1)
	go func() {
		err := instance.Start()
		if err != nil {
			errChan <- err
			logger.Error("beaver fail", log.Error(err))
		} else {
			okChan <- struct{}{}
		}
	}()

	stopChan := make(chan os.Signal, 1)
	signal.Notify(stopChan, syscall.SIGINT, syscall.SIGTERM)
	select {
	case <-stopChan:
		logger.Info("shutting down gracefully by signal")

		ctx, cancel := context.WithTimeout(context.Background(), config.ShutdownDeadline)
		defer cancel()

		if err := instance.Shutdown(ctx); err != nil {
			logger.Error("failed to shutdown beaver", log.Error(err))
		}
	case <-okChan:
		return
	case <-errChan:
		return
	}
}
