package cleaner

import (
	"context"
	"fmt"
	"strconv"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/gideon/speedy-beaver/internal/config"
	"a.yandex-team.ru/security/gideon/speedy-beaver/internal/db"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
	"a.yandex-team.ru/yt/go/yterrors"
	"a.yandex-team.ru/yt/go/ytlock"
)

const (
	lockName = "clean-lock"
)

type Cleaner struct {
	yt  yt.Client
	cfg config.Config
	log log.Logger
}

func NewCleaner(cfg config.Config, opts ...Option) (*Cleaner, error) {
	if cfg.Cleaner.TTL <= 1 {
		return nil, fmt.Errorf("ttl is too small: %d <= 1", cfg.Cleaner.TTL)
	}

	c := &Cleaner{
		cfg: cfg,
		log: &nop.Logger{},
	}

	for _, opt := range opts {
		opt(c)
	}

	var err error
	c.yt, err = ythttp.NewClient(&yt.Config{
		Proxy:  cfg.Cleaner.YtProxy,
		Token:  cfg.Cleaner.YtToken,
		Logger: c.log.Structured(),
	})

	if err != nil {
		return nil, fmt.Errorf("can't create YT client: %w", err)
	}

	return c, nil
}

func (c *Cleaner) Start(ctx context.Context) error {
	clean := func() {
		err := c.Cleanup(ctx)
		if err != nil {
			c.log.Error("can't cleanup old partitions", log.Error(err))
		}
	}

	clean()

	for {
		select {
		case <-ctx.Done():
			return nil
		case <-time.After(c.cfg.Cleaner.CheckTick):
			clean()
		}
	}
}

func (c *Cleaner) Cleanup(ctx context.Context) error {
	lock := ytlock.NewLockOptions(c.yt, ypath.Path(c.cfg.Cleaner.YtPath).JoinChild(lockName), ytlock.Options{
		CreateIfMissing: true,
		LockMode:        yt.LockExclusive,
	})

	_, err := lock.Acquire(ctx)
	if err != nil {
		if yterrors.ContainsErrorCode(err, yterrors.CodeConcurrentTransactionLockConflict) {
			c.log.Info("conflict YT lock", log.String("error", err.Error()))
			return nil
		}
		return fmt.Errorf("failed to acquire lock: %w", err)
	}
	defer func() { _ = lock.Release(ctx) }()

	c.log.Info("YT lock acquired, try to cleanup")

	systemDB, err := db.NewSystemDB(c.cfg.ClickHouse)
	if err != nil {
		return fmt.Errorf("failed to create system DB: %w", err)
	}
	defer func() { _ = systemDB.Close(ctx) }()

	parts, err := systemDB.ListPartitions(context.Background())
	if err != nil {
		return fmt.Errorf("failed to list gideon partitions: %w", err)
	}

	maxPart, _ := strconv.Atoi(time.Now().Add(-c.cfg.Cleaner.TTL).Format("20060102"))
	for _, p := range parts {
		if p > maxPart {
			continue
		}

		err = systemDB.DropPartition(ctx, p)
		if err != nil {
			c.log.Error("can't drop partition", log.Int("partition", p), log.Error(err))
			continue
		}

		c.log.Error("partition dropped", log.Int("partition", p))
	}

	return nil
}
