package config

import (
	"context"
	"fmt"
	"net/url"
	"os"
	"strings"
	"time"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/file"

	"a.yandex-team.ru/library/go/core/log"
)

const (
	ShutdownDeadline = 10 * time.Second
)

type Logger struct {
	Level string `yaml:"level"`
}

type Logbroker struct {
	Endpoints []string `yaml:"endpoints"`
	Consumer  string   `yaml:"consumer"`
	Topics    []string `yaml:"topics"`
}

type ClickHouse struct {
	Addrs     []string      `yaml:"addrs"`
	Database  string        `yaml:"database"`
	User      string        `yaml:"user"`
	Password  string        `yaml:"password"`
	Compress  bool          `yaml:"compress"`
	Debug     bool          `yaml:"debug"`
	Retries   uint64        `yaml:"retries"`
	BatchSize int           `yaml:"batch_size"`
	BatchTick time.Duration `yaml:"batch_tick"`
}

type Cleaner struct {
	YtProxy   string        `yaml:"yt_proxy"`
	YtPath    string        `yaml:"yt_path"`
	YtToken   string        `yaml:"yt_token"`
	TTL       time.Duration `yaml:"ttl"`
	CheckTick time.Duration `yaml:"check_tick"`
}

type Lbmask struct {
	Endpoint string `yaml:"endpoint"`
	Topic    string `yaml:"topic"`
}

type Config struct {
	Debug      bool       `yaml:"debug"`
	Logger     Logger     `yaml:"logger"`
	Logbroker  Logbroker  `yaml:"logbroker"`
	ClickHouse ClickHouse `yaml:"clickhouse"`
	Cleaner    Cleaner    `yaml:"cleaner"`
	Lbmask     Lbmask     `yaml:"lbmask"`
}

func NewConfig(configPath string) (*Config, error) {
	cfg := &Config{
		Debug: false,
		Logger: Logger{
			Level: log.InfoString,
		},
		Logbroker: Logbroker{
			Endpoints: []string{
				"man.logbroker.yandex.net",
				"myt.logbroker.yandex.net",
				"sas.logbroker.yandex.net",
				"vla.logbroker.yandex.net",
				"iva.logbroker.yandex.net",
			},
		},
		ClickHouse: ClickHouse{
			Database:  "gideon",
			Password:  os.Getenv("CH_PASSWD"),
			Compress:  true,
			BatchSize: 1024,
			Retries:   2,
			BatchTick: 1 * time.Second,
		},
		Cleaner: Cleaner{
			TTL:       3 * 24 * time.Hour,
			YtProxy:   "locke",
			YtPath:    "//home/security/speedy-beaver/dev",
			YtToken:   os.Getenv("YT_TOKEN"),
			CheckTick: 1 * time.Hour,
		},
	}

	loader := confita.NewLoader(file.NewBackend(configPath))
	if err := loader.Load(context.Background(), cfg); err != nil {
		return nil, fmt.Errorf("failed to parse config: %w", err)
	}
	return cfg, nil
}

func (c *ClickHouse) mergeOpts(opts url.Values) url.Values {
	if opts == nil {
		opts = url.Values{}
	}

	opts.Set("database", c.Database)
	opts.Set("username", c.User)
	opts.Set("password", c.Password)

	if c.Compress {
		opts.Set("compress", "true")
	}

	if c.Debug {
		opts.Set("debug", "true")
	}

	return opts
}

func (c *ClickHouse) URI(opts url.Values) string {
	opts = c.mergeOpts(opts)
	if len(c.Addrs) > 1 {
		opts.Set("alt_hosts", strings.Join(c.Addrs[1:], ","))
	}
	uri := fmt.Sprintf("tcp://%s?%s", c.Addrs[0], opts.Encode())

	if len(c.Addrs) > 1 {
		uri = fmt.Sprintf("%s&alt_hosts=%s", uri, strings.Join(c.Addrs[1:], ","))
	}
	return uri
}

func (c *ClickHouse) AddrURI(addr string, opts url.Values) string {
	opts = c.mergeOpts(opts)
	return fmt.Sprintf("tcp://%s?%s", addr, opts.Encode())
}
