import React, {MouseEvent, PropsWithChildren} from "react";
import Moment from "moment";
import {faExternalLinkSquareAlt} from "@fortawesome/free-solid-svg-icons";
import {FontAwesomeIcon} from "@fortawesome/react-fontawesome";
import {Button} from "lego-on-react";
import {
    Event,
    EventConnect,
    EventKind,
    EventOpenAt,
    EventProcExec,
    EventPtrace,
    EventSSHSession,
    SSHSessionKind
} from "../../models/gideon";

import classes from "./EventsTable.module.css";

interface Props {
    events: Event[];
}

type EventGroups = { [name: string]: Event[] };

const groupEvents = (inEvents: Event[]): { keys: string[]; events: EventGroups } => {
    let keys: string[] = [];
    let events: EventGroups = inEvents.reduce(function (r: EventGroups, e: Event) {
        const key = e.source + "-" + e.host + "-" + e.podSetID + "-" + e.podID + "-" + String(e.sessionID) + "-" + e.container;
        if (!r[key]) {
            keys.push(key);
            r[key] = [e];
        } else {
            r[key].push(e);
        }

        return r;
    }, {});

    return {keys, events};
};

const renderEventsHeader = (event: Event) => {
    const podSetOrNanny = (event: Event) => {
        if (event.nannyServiceID !== undefined && event.nannyServiceID !== "") {
            return (
                <React.Fragment>
                    NannyServiceID: <strong>{event.nannyServiceID}</strong>
                </React.Fragment>
            );
        }

        return (
            <React.Fragment>
                PodSetID: <strong>{event.podSetID}</strong>
            </React.Fragment>
        );
    };

    return (
        <React.Fragment>
            Source: <strong>{event.source}</strong> Host: <strong>{event.host}</strong> Container:{" "}
            <strong>{event.container}</strong> {podSetOrNanny(event)} PodID: <strong>{event.podID}</strong> SessionID:{" "}
            <strong>{event.sessionID}</strong>
        </React.Fragment>
    );
};

const renderEvent = (kind: string, event: Event) => (
    <React.Fragment>
        <td>{Moment(event.time).format("DD.MM HH:mm:ss")}</td>
        <td style={{textAlign: "right"}}>{event.ppid}{event.parentName ? ` (${event.parentName})` : ""}</td>
        <td style={{textAlign: "right"}}>{event.pid}{event.name ? ` (${event.name})` : ""}</td>
        <td style={{textAlign: "right"}}>{event.uid}</td>
        <td style={{textAlign: "right"}}>{kind}</td>
    </React.Fragment>
);

const renderProcExec = (event: EventProcExec) => (
    <React.Fragment>
        {renderEvent("ProcExec", event)}
        <td style={{textAlign: "left"}}>
            <code>
                {event.exe} {event.args.map((v) => "'" + v + "'").join(" ")}
            </code>
        </td>
    </React.Fragment>
);

const renderConnect = (event: EventConnect) => (
    <React.Fragment>
        {renderEvent("Connect", event)}
        <td style={{textAlign: "left"}}>
            <code>
                [{event.dstAddr}]:{event.dstPort}
            </code>
        </td>
    </React.Fragment>
);

const ptraceRequest = (req: number): string => {
    if (req === 16) {
        return "attach";
    }

    if (req === 17) {
        return "detach";
    }

    return `unknown_${req}`;
};

const sshSessionKind = (kind: SSHSessionKind): string => {
    if (kind == SSHSessionKind.SSH) {
        return "ssh";
    }

    if (kind == SSHSessionKind.Portoshell) {
        return "portoshell";
    }

    if (kind == SSHSessionKind.YtJobshell) {
        return "yt-jobshell";
    }

    return `unknown_${kind}`;
}

const renderPtrace = (event: EventPtrace) => (
    <React.Fragment>
        {renderEvent("Ptrace", event)}
        <td style={{textAlign: "left"}}>
            <code>
                req={ptraceRequest(event.request)} pid={event.targetPID} ret={event.retCode}
            </code>
        </td>
    </React.Fragment>
);

const openAtFD = (fd: number): string => {
    if (fd === 4294967196 || fd === -100) {
        return "AT_FDCWD";
    }

    return String(fd);
};

const openAtFlags = (flag: number): string => {
    enum Flags {
        O_RDONLY = 0x0,
        O_WRONLY = 0x1,
        O_RDWR = 0x2,
        O_APPEND = 0x400,
        O_CREAT = 0x40,
        O_TRUNC =  0x200,
    }

    let out: string[] = [];
    for (let a in Flags) {
        const val = Number(a);
        if (isNaN(val)) {
            continue;
        }

        if (flag&val) {
            out.push(Flags[val])
        }
    }

    return String(out.join("|"));
};

const renderOpenAt = (event: EventOpenAt) => (
    <React.Fragment>
        {renderEvent("OpenAt", event)}
        <td style={{textAlign: "left"}}>
            <code>
                fd={openAtFD(event.fd)} filename={event.filename} flags={openAtFlags(event.flags)}
            </code>
        </td>
    </React.Fragment>
);

const renderSSHSession = (event: EventSSHSession) => {
    const onClick = (e: MouseEvent) => {
        e.preventDefault();

        const sessionInfo = {
            TS: event.ts - 2e9,
            PodID: event.podID,
            SessionID: event.sessionID,
            Source: event.source,
            Host: event.host,
        }

        const key = btoa(JSON.stringify(sessionInfo)).replace(/\+/g, '-').replace(/\//g, '_');
        window.open(`/session#key=${key}`, "_blank");
    };

    return <React.Fragment>
        {renderEvent("NewSession", event)}
        <td style={{textAlign: "left"}}>
            <code>
                kind={sshSessionKind(event.sshKind)} id={event.sshID} tty={event.sshTTY} user={event.sshUser} <Button theme="clear"
                                                                                 onMouseDown={onClick}><FontAwesomeIcon
                icon={faExternalLinkSquareAlt}/></Button>
            </code>
        </td>
    </React.Fragment>
};

export const EventsTable = ({events}: PropsWithChildren<Props>) => {
    if (!events || events.length === 0) {
        return (
            <div style={{textAlign: "center"}}>
                There are no results.
                <br/>
                Try to change a search query (e.g. <code>time&gt;=-1h; host=my.cool.host.yandex.net;
                pod_set_id~=ps-%;</code>).
                <br/>
            </div>
        );
    }

    const eventsGroups = groupEvents(events);
    return (
        <React.Fragment>
            <table className={classes["events-table"]}>
                <thead>
                <tr>
                    <th style={{textAlign: "left"}}>Time</th>
                    <th style={{textAlign: "right"}}>Parent</th>
                    <th style={{textAlign: "right"}}>Current</th>
                    <th style={{textAlign: "right"}}>Uid</th>
                    <th style={{textAlign: "right"}}>Kind</th>
                    <th style={{textAlign: "left", width: "70%"}}>Details</th>
                </tr>
                </thead>
                <tbody>
                {eventsGroups.keys.map((key) => {
                    const events = eventsGroups.events[key];
                    return (
                        <React.Fragment key={key}>
                            <tr>
                                <td colSpan={6}>{renderEventsHeader(events[0])}</td>
                            </tr>
                            {events.map((event, i) => (
                                <tr key={i}>
                                    {event.kind === EventKind.ProcExec && renderProcExec(event as EventProcExec)}
                                    {event.kind === EventKind.Connect && renderConnect(event as EventConnect)}
                                    {event.kind === EventKind.PTrace && renderPtrace(event as EventPtrace)}
                                    {event.kind === EventKind.OpenAt && renderOpenAt(event as EventOpenAt)}
                                    {event.kind === EventKind.SSHSession && renderSSHSession(event as EventSSHSession)}
                                </tr>
                            ))}
                        </React.Fragment>
                    );
                })}
                </tbody>
            </table>
        </React.Fragment>
    );
};

EventsTable.displayName = "EventsTable";
