import {Position, Provider} from "@yandex-infracloud-ui/query-input";
import {Lexer} from "chevrotain";
import {editor as monacoEditor, languages, Position as monacoPosition} from "monaco-editor/esm/vs/editor/editor.api";

import {GideonApi} from "../../../../services/api/GideonApi";
import {Filter, FilterOperator} from "../../../../models/gideon";

import {CurrentTokenInterpreter} from "./CurrentTokenInterpreter";
import {queryParser} from "./QueryParser";
import {QueryInterpreter} from "./QueryInterpreter";
import {RequestInterpreter} from "./RequestInterpreter";
import tokens from "./tokens";

import throttle from "lodash-es/throttle";

const QueryLexer = new Lexer(tokens);

export const suggestionKeys = [
    "kind",
    "source",
    "host",
    "pod_id",
    "pod_set_id",
    "nanny_service_id",
    "session_id",
    "ssh_session_id",
    "ssh_session_kind",
    "time",
];
export const eqKeys = ["kind", "source", "host", "pod_id", "pod_set_id", "nanny_service_id", "session_id", "ssh_session_id", "ssh_session_kind"];
export const gtKeys = ["time"];
export const ltKeys = ["time"];

export const kindValues = ["ProcExec", "Connect", "Ptrace", "OpenAt", "NewSession"];
export const sshSessionKindValues = ["ssh", "portoshell", "yt_jobshell"];
export const timeValues = ["-12h", "-1d", "1614248032", "2006-01-02T15:04"];

const serializer = new QueryInterpreter();
const requester = new RequestInterpreter();
const tokenator = new CurrentTokenInterpreter();

type requestItem = {
    selectType: string;
    values: string[];
};

type requestFilter = {[key: string]: requestItem};

export class EventsProvider implements Provider {
    private gideonApi: GideonApi;
    private loadMoreCommandId: string | null = null;
    private isFullSearch = false;
    private cache = new Map<string, Promise<languages.CompletionItem[]>>();

    static isContextSuggest(query: string, position: Position) {
        return false;
    }

    public initializeEditor(editor: monacoEditor.IStandaloneCodeEditor) {
        this.loadMoreCommandId = editor.addCommand(0, () => {
            this.isFullSearch = true;
            setTimeout(() => editor.trigger("suggest", "editor.action.triggerSuggest", ""));
        });
    }

    constructor(gideonApi: GideonApi) {
        this.gideonApi = gideonApi;
    }

    public async onSuggest(
        model: monacoEditor.ITextModel,
        position: monacoPosition
    ): Promise<languages.CompletionItem[]> {
        const inputString = model.getValue();

        // const suggestHash = `${inputString}_${position.column}_${position.lineNumber}_${this.isFullSearch}`;
        // const cachedResult = this.cache.get(suggestHash);

        // if (cachedResult) {
        //     return cachedResult;
        // }

        queryParser.input = QueryLexer.tokenize(inputString).tokens;
        const query = queryParser.query();
        const token = tokenator.visit(query, position);
        const ast = serializer.visit(query) || {};

        if (!inputString || !query || !token || token.type === "KEY") {
            const currentToken = queryParser.getToken(position);
            return this.onSuggestKeys(currentToken, position, ast);
        }

        if (token && token.type === "VALUE") {
            if (token.key == "ssh_session_kind") {
                return candidateToSuggest(token, sshSessionKindValues);
            }

            if (token.key === "kind") {
                return candidateToSuggest(token, kindValues);
            }

            if (token.key === "time") {
                if (token.value.length > 0) {
                    return [] as languages.CompletionItem[];
                }

                return candidateToSuggest(token, timeValues);
            }

            return [] as languages.CompletionItem[];
            // const suggestValuesResult = this.onSuggestValues(token, position, query);
            // this.cache.clear();
            // this.cache.set(suggestHash, suggestValuesResult);
            // return suggestValuesResult;
        }

        return [] as languages.CompletionItem[];
    }

    private onSuggestKeys(token: any, position: Position, ast: any) {
        return [
            ...eqKeys
                .filter((key) => !ast.hasOwnProperty(key))
                .map((item) => ({
                    insertText: item + "=",
                    kind: languages.CompletionItemKind.Text,
                    label: item + "=",
                })),
            ...gtKeys
                .filter((key) => !ast.hasOwnProperty(key))
                .map((item) => ({
                    insertText: item + ">=",
                    kind: languages.CompletionItemKind.Text,
                    label: item + ">=",
                })),
            ...ltKeys
                .filter((key) => !ast.hasOwnProperty(key))
                .map((item) => ({
                    insertText: item + "<=",
                    kind: languages.CompletionItemKind.Text,
                    label: item + "<=",
                })),
        ] as languages.CompletionItem[];
    }

    private async onSuggestValues(token: any, position: Position, query: any) {
        const throttled = throttle(async (token: any, position: Position, query: any) => {
            const {key, value} = token;

            if (suggestionKeys.indexOf(key) === -1) {
                return [];
            }

            const valuePrefix = value.slice(0, position.column - token.startColumn);
            const request: requestFilter = requester.visit(query) || {};
            delete request[key + "_list"];

            const fullSearch = this.isFullSearch;
            this.isFullSearch = false;

            let filter: Filter = [];
            for (let [k, e] of Object.entries(request)) {
                const filterKey = k.replace("_list", "");
                if (key === "userFieldList" || filterKey === "userFieldList" || filterKey === key) {
                    continue;
                }

                filter.push({
                    key: filterKey,
                    operator: (FilterOperator as any)[e.selectType],
                    values: e.values,
                });
            }

            const values = await this.gideonApi
                .suggestValue({
                    fullSearch,
                    key: key,
                    filter: filter,
                    valuePrefix,
                })
                .toPromise();

            if (!values || values.length === 0) {
                return [];
            }

            const result: any[] = candidateToSuggest(token, values);
            if (!fullSearch && this.loadMoreCommandId) {
                result.push({
                    filterText: " " + valuePrefix,
                    insertText: "",
                    label: "load more...",
                    preselect: false,
                    sortText: "z",
                    kind: languages.CompletionItemKind.Function,

                    range: {
                        endColumn: position.column,
                        endLineNumber: position.lineNumber,
                        startColumn: position.column,
                        startLineNumber: position.lineNumber,
                    },

                    command: {
                        id: this.loadMoreCommandId,
                        title: "loadMore",
                    },
                });
            }

            return result as languages.CompletionItem[];
        }, 1000);

        return (await throttled(token, position, query)) || [];
    }
}

const candidateToSuggest = (token: any, candidates: string[]): languages.CompletionItem[] => {
    return candidates.map((item) => {
        if (!/^[^,=;\s]*$/.test(item)) {
            // reserved symbols must be wrapped
            item = `"${item}"`;
        }

        let insertText = item;
        let label = item;

        if (item === "") {
            label = "<EMPTY>";
            insertText = '""';
        }

        insertText += "; ";

        return {
            insertText,
            filterText: item,
            kind: languages.CompletionItemKind.Text,
            label,
            range:
                token.startColumn !== token.endColumn
                    ? {
                          endColumn: token.endColumn + 2,
                          endLineNumber: token.endLine,
                          startColumn: token.startColumn,
                          startLineNumber: token.startLine,
                      }
                    : undefined,
        };
    }) as languages.CompletionItem[];
};
