import React, {PropsWithChildren} from "react";
import Moment from "moment";
import {
    Event,
    EventProcExec,
    EventConnect,
    EventPtrace,
    EventSSHSession,
    EventKind,
    EventOpenAt,
    SSHSessionKind
} from "../../models/gideon";
import archy from "../../utils/archy";

import classes from "./SessionView.module.css";

interface Props {
    events: Event[];
}

type EventNode = Event & {
    childs: EventNode[];
}

const checkSameSession = (inEvents: Event[]): boolean => {
    let session_key = "";
    for (let e of inEvents) {
        const key = e.source + "-" + e.host + "-" + e.podSetID + "-" + e.podID + "-" + String(e.sessionID) + "-" + e.container;
        if (session_key.length === 0) {
            session_key = key;
            continue;
        }

        if (session_key !== key) {
            return false;
        }
    }

    return true;
};

const renderEventPrefix = (event: Event) => (
    `${String(event.uid).padEnd(10)}${String(event.ppid).padEnd(10)}${String(event.pid).padEnd(10)}${Moment(event.time).format("HH:mm:ss").padEnd(15)}`
);

const renderProcExec = (event: EventProcExec) => (
    `exec: ${event.exe} ${event.args.map((v) => `'${v}'`).join(" ")}`
);

const renderConnect = (event: EventConnect) => (
    `connect: [${event.dstAddr}]:${event.dstPort}`
);

const ptraceRequest = (req: number): string => {
    if (req === 16) {
        return "attach";
    }

    if (req === 17) {
        return "detach";
    }

    return `unknown_${req}`;
};

const sshSessionKind = (kind: SSHSessionKind): string => {
    if (kind == SSHSessionKind.SSH) {
        return "ssh";
    }

    if (kind == SSHSessionKind.Portoshell) {
        return "portoshell";
    }

    if (kind == SSHSessionKind.YtJobshell) {
        return "yt-jobshell";
    }

    return `unknown_${kind}`;
}

const renderPtrace = (event: EventPtrace) => (
    [
        `ptrace`,
        `req: ${ptraceRequest(event.request)}`,
        `pid: ${event.targetPID}`,
        `ret: ${event.retCode}`,
    ].join('\n')
);


const openAtFD = (fd: number): string => {
    if (fd === 4294967196 || fd === -100) {
        return "AT_FDCWD";
    }

    return String(fd);
};

const openAtFlags = (flag: number): string => {
    enum Flags {
        O_RDONLY = 0x0,
        O_WRONLY = 0x1,
        O_RDWR = 0x2,
        O_APPEND = 0x400,
        O_CREAT = 0x40,
        O_TRUNC =  0x200,
    }

    let out: string[] = [];
    for (let a in Flags) {
        const val = Number(a);
        if (isNaN(val)) {
            continue;
        }

        if (flag&val) {
            out.push(Flags[val])
        }
    }

    return String(out.join("|"));
};

const renderOpenAt = (event: EventOpenAt) => (
    `openat(${openAtFD(event.fd)}, "${event.filename}" ${openAtFlags(event.flags)})`
);

const renderError = (msg: string) => (
    <React.Fragment>
        <div style={{textAlign: "center"}}>
            Something goes wrong: {msg}.
            <br />
            Please report to <code>security@yandex-team.ru</code>.
            <br />
        </div>
    </React.Fragment>
)

const renderSSHSession = (event: EventSSHSession) => {
    const podSetOrNanny = (event: Event) => {
        if (event.nannyServiceID !== undefined && event.nannyServiceID !== "") {
            return `nanny service: ${event.nannyServiceID}`;
        }

        if (event.podSetID !== undefined && event.podSetID !== "") {
            return `podset id: ${event.podSetID}`;
        }

        return "";
    };

    const eventTime = Moment(event.time);
    return [
        'session started',
        `timestamp: ${eventTime.unix()}`,
        `time: ${eventTime.format("DD.MM.YYYY HH:mm:ss")}`,
        `host: ${event.host}`,
        `kind: ${sshSessionKind(event.sshKind)}`,
        podSetOrNanny(event),
        event.podID ? `pod id: ${event.podID}` : "",
        `gideon id: ${event.sessionID}`,
        `ssh id: ${event.sshID}`,
        `user: ${event.sshUser}`,
        event.container ? `container: ${event.container}` : "",
    ].filter(i => !!i).join('\n');
};

const treeifyEvents = (events: Event[]): EventNode[] => {
    let tree = [];
    let childrenOf: { [key: string]: EventNode[]} = {};

    const key = (kind: number, pid: number) => `${kind}.${pid}`;
    for (const e of events) {
        const id = key(e.kind, e.pid);
        const parentId = e.kind === EventKind.ProcExec ? key(EventKind.ProcExec, e.ppid): key(EventKind.ProcExec, e.pid);
        childrenOf[id] = childrenOf[id] || [];

        let item = {...e, childs: childrenOf[id] || []};
        if (parentId.length > 0 && parentId in childrenOf) {
            childrenOf[parentId].push(item);
        } else {
            tree.push(item);
        }
    };

    return tree;
}

export const SessionView = ({events}: PropsWithChildren<Props>) => {
    if (!events || events.length === 0) {
        return (
            <div style={{textAlign: "center"}}>
                There are no results.
            </div>
        );
    }

    for (let i = 0; i < events.length; i++) {
        if (events[0].kind !== EventKind.SSHSession) {
            events.shift();
        }
    }

    if (events.length === 0 || events[0].kind !== EventKind.SSHSession) {
        return renderError("no initial session info");
    }

    if (!checkSameSession(events)) {
        return renderError("multiple sessions");
    }

    const rootNode = {
        ...events.shift(),
        childs: treeifyEvents(events),
    }

    const renderedEvents = archy.draw(
        rootNode,
        {
            drawRootBranch: false,
            nodes: "childs",
            prefix: (e: EventNode) => renderEventPrefix(e),
            label: (e: EventNode) => {
                if (e.kind === EventKind.SSHSession) {
                    return renderSSHSession(e as EventSSHSession);
                }

                if (e.kind === EventKind.ProcExec) {
                    return renderProcExec(e as EventProcExec);
                }

                if (e.kind === EventKind.Connect) {
                    return renderConnect(e as EventConnect);
                }

                if (e.kind === EventKind.PTrace) {
                    return renderPtrace(e as EventPtrace);
                }

                if (e.kind === EventKind.OpenAt) {
                    return renderOpenAt(e as EventOpenAt);
                }

                return "ooops, some shit :(";
            },
        },
    );

    return (
        <React.Fragment>
            <pre className={classes["session"]}>
                {"UID".padEnd(10)}
                {"PPID".padEnd(10)}
                {"PID".padEnd(10)}
                {"TIME".padEnd(15)}
                {"EVENT\n"}
                {renderedEvents}
            </pre>
        </React.Fragment>
    );
};

SessionView.displayName = "SessionView";
