import {faSearch} from "@fortawesome/pro-regular-svg-icons";
import {FontAwesomeIcon} from "@fortawesome/react-fontawesome";
import {classNames, fixedMemo, IPureValueProps, Keys, LegoTextInput} from "@yandex-infracloud-ui/libs";
import React, {useCallback, useRef, useState} from "react";

import classes from "./FastSearchInput.module.css";

interface Props extends IPureValueProps<string> {
    className?: string;
    name: string;
    placeholder?: string;
    forceVisible?: boolean;
}

export const FastSearchInput: React.FC<Props> = fixedMemo(
    ({className, name, placeholder = "Fast filter...", forceVisible, value, onChange}) => {
        const innerRef = useRef<HTMLInputElement>();

        const [visible, setVisible] = useState(forceVisible ?? Boolean(value));

        const show = useCallback(() => setVisible(true), []);

        const handleKeyDown = useCallback(
            (e: React.KeyboardEvent<HTMLInputElement>) => {
                switch (e.key) {
                    case Keys.Escape:
                        onChange("");
                        setVisible(false);
                        break;
                }
            },
            [onChange]
        );

        const handleBlur = useCallback(() => {
            if (!value) {
                setVisible(forceVisible ?? false);
            }
        }, [forceVisible, value]);

        const setInnerRef = useCallback((input: HTMLInputElement) => {
            innerRef.current = input;
        }, []);

        return (
            <LegoTextInput
                cls={classNames(classes.input, className, {[classes.inputVisible]: visible || value})}
                hasClear={true}
                icon={<FontAwesomeIcon className={classes.inputIcon} icon={faSearch} />}
                innerRef={setInnerRef}
                name={name}
                onBlur={handleBlur}
                onChange={onChange}
                onFocus={show}
                onKeyDown={handleKeyDown}
                placeholder={placeholder}
                text={value}
            />
        );
    }
);

FastSearchInput.displayName = "FastSearchInput";
