import React, {useRef, useState, useEffect} from "react";
import {fixedMemo, useDismounted} from "@yandex-infracloud-ui/libs";
import {useQueryParam, StringParam} from "use-query-params";
import {Button} from "lego-on-react";
import {map, finalize, takeUntil} from "rxjs/operators";
import {Loader} from "@yandex-infracloud-ui/common";

import {EventsQueryInput, EventsProvider} from "../../components/QueryInput";
import {EventsTable} from "../../components/EventsTable/EventsTable";
import {useGideonApi} from "../../services";
import {Event} from "../../models/gideon";

import {parseQuery, ParsedQuery} from "./parse";
import classes from "./Events.module.css";

const filterFields: {[key: string]: string} = {
    source: "source",
    kind: "kind",
    host: "host",
    pod_id: "pod_id",
    pod_set_id: "pod_set_id",
    nanny_service_id: "nanny_service_id",
    session_id: "session_id",
    ssh_session_id: "ssh_session_id",
    ssh_session_kind: "ssh_session_kind",
    time: "time",
};

const getFetchQueryParams = (query: string): ParsedQuery => {
    let result = parseQuery(query);

    result.filter.forEach((e) => {
        if (!filterFields[e.key]) {
            result.errors.push(`'${e.key}' is unknown parameter. `);
        }
    });

    return result;
};

export const Events = fixedMemo(() => {
    const gideonApi = useGideonApi();
    const dismounted = useDismounted();

    const [isLoading, setIsLoading] = useState(false);
    const [loadError, setLoadError] = useState("");
    const [events, setEvents] = useState<Event[]>([]);
    const [query, setQuery] = useQueryParam("filter", StringParam);
    const [finalQuery, setFinalQuery] = useState(query || "");
    const parsedQuery = getFetchQueryParams(query || "");

    useEffect(() => {
        const parsedQuery = getFetchQueryParams(finalQuery);
        if (parsedQuery.errors.length > 0) {
            return;
        }

        if (Object.keys(parsedQuery.filter).length === 0) {
            return;
        }

        setLoadError("");
        setIsLoading(true);
        gideonApi
            .fetchEvents(parsedQuery.filter)
            .pipe(
                map((events) => setEvents(events)),
                finalize(() => setIsLoading(false)),
                takeUntil(dismounted)
            )
            .subscribe({
                error: (err: Error) => setLoadError(err.message),
            });
    }, [dismounted, gideonApi, finalQuery]);

    const onFiltersReset = () => {
        setQuery("");
        setFinalQuery("");
    };

    const onFiltersSubmit = () => {
        if (parsedQuery.errors.length > 0) {
            return;
        }

        setFinalQuery(query || "");
    };

    const onInputChange = (value?: string) => {
        const v = value ? value.trim() : "";
        setQuery(v);
    };

    const onSuggestLinkClick = (value: string) => {
        // TODO(buglloc): WTF?!
        if (value === "time") {
            setQuery(`${value} >= 2006-01-02T15:04; `);
            return;
        }

        setQuery(`${value} = kek, cheburek; `);
    };

    const e = useRef<EventsProvider | undefined>(undefined);
    if (e.current === undefined) {
        e.current = new EventsProvider(gideonApi);
    }

    const renderFilters = () => {
        return (
            <React.Fragment>
                <div className={classes["filter-container"]}>
                    <div className={classes["filter-query"]}>
                        <EventsQueryInput
                            query={query || ""}
                            onChange={onInputChange}
                            {...(e.current ? {provider: e.current} : {})}
                        />
                    </div>
                    <div className={classes["filter-controls"]}>
                        <Button
                            key={"filters-reset"}
                            theme={"pseudo"}
                            size={"s"}
                            onClick={onFiltersReset}
                            width={"max"}
                        >
                            Reset
                        </Button>
                        <Button
                            key={"filters-submit"}
                            size={"s"}
                            onClick={onFiltersSubmit}
                            width={"max"}
                            theme={"action"}
                            disabled={parsedQuery.errors.length > 0}
                        >
                            Filter
                        </Button>
                    </div>
                </div>
                {parsedQuery.errors.length > 0 && (
                    <span className={classes["filter-error"]}>{parsedQuery.errors.join(" ")}</span>
                )}
                {"Search by keys: "}
                {Object.keys(filterFields).map((field, index) => (
                    <React.Fragment key={`key--suggest-link-${field}`}>
                        {index > 0 ? ", " : ""}
                        <span className={classes.link} onClick={() => onSuggestLinkClick(field)}>
                            {field}
                        </span>
                    </React.Fragment>
                ))}
                {". "}
            </React.Fragment>
        );
    };

    return (
        <div className={classes["events_view"]}>
            {renderFilters()}
            <div className={classes["events-body"]}>
                {(loadError && <span className={classes["events-error"]}>Ooops: {loadError}</span>) ||
                    (isLoading && (
                        <div className={classes["loader-container"]}>
                            <Loader size="l" />
                        </div>
                    )) || <EventsTable events={events}></EventsTable>}
            </div>
        </div>
    );
});
