import {Filter, FilterOperator} from "../../models/gideon";

export type ParsedQuery = {
    filter: Filter;
    errors: string[];
};

const gideonOperators: Map<string, FilterOperator> = new Map([
    ["=", FilterOperator.EQ],
    ["!=", FilterOperator.NE],
    ["~=", FilterOperator.EL],
    ["!~=", FilterOperator.NL],
    [">=", FilterOperator.GT],
    ["<=", FilterOperator.LT],
]);

export const parseQuery = (query: string): ParsedQuery => {
    let out: ParsedQuery = {
        filter: [],
        errors: [],
    };

    let key = "";
    let value = "";
    let include = "";
    // always start with a key!
    let isKey = true;
    let isValue = false;
    let isLiteralValue = false;
    let isLiteralValueClosed = false;
    let values = [];

    // incorrect queries:
    // message ==!!== abc
    // mess age = a b
    // message a b
    // message = a b
    // message = "a ;
    // message = b " c "
    // message = " c " d
    // message = " d " " e "
    // message = e , ""  f  ""
    // message = , , ,

    if (query && query.trim()) {
        for (let i = 0, l = query.length; i < l; i++) {
            // service character inside a literal querying
            if (
                isValue &&
                isLiteralValue &&
                !isLiteralValueClosed &&
                ["=", " ", "\n", ";", ",", "!", "<", ">", "~"].includes(query[i])
            ) {
                value += query[i];
                // start or an end of a literal querying value
            } else if (query[i] === '"' && query[i - 1] !== "\\") {
                if (isLiteralValueClosed) {
                    out.errors = [
                        "Search query is incorrect. ",
                        `Values error (${value}${query[i]}). `,
                        "Commas or double quotes are required. ",
                    ];
                    return out;
                } else if (isLiteralValue) {
                    isValue = false;
                    isLiteralValueClosed = true;
                } else {
                    if (isValue && value.trim()) {
                        out.errors = [
                            "Search query is incorrect. ",
                            `Values error (${value}${query[i]}). `,
                            "Commas or double quotes are required. ",
                        ];
                        return out;
                    } else {
                        isValue = true;
                        isLiteralValue = true;
                        isLiteralValueClosed = false;
                    }
                }
                // include value
            } else if (query[i] === "!" || query[i] === "~" || query[i] === "=" || query[i] === ">" || query[i] === "<") {
                isKey = false;
                include += query[i];
                // guaranteed end of a value
            } else if (query[i] === ";") {
                key = key.trim();

                if (key) {
                    if (key.includes(" ") || key.includes("\n")) {
                        out.errors = ["Search query is incorrect. ", `Key error (${key}). `];
                        return out;
                    }

                    if (isLiteralValue || (!isLiteralValue && value.trim())) {
                        if (isLiteralValue && !isLiteralValueClosed) {
                            out.errors = [
                                "Search query is incorrect. ",
                                `Values error (${value}). `,
                                "Commas or double quotes are required. ",
                            ];
                            return out;
                        } else if (!isLiteralValue && (value.trim().includes(" ") || value.trim().includes("\n"))) {
                            out.errors = [
                                "Search query is incorrect. ",
                                `Values error (${value}). `,
                                "Commas or double quotes are required. ",
                            ];
                            return out;
                        }

                        values.push(isLiteralValue ? value : value.trim());
                    } else {
                        out.errors = ["Search query is incorrect. ", "Value error. "];
                        return out;
                    }

                    if (!gideonOperators.has(include)) {
                        out.errors = ["Search query is incorrect. ", `Syntax error: "${include}". `];
                        return out;
                    }
                    
                    out.filter.push({
                        key: key,
                        values,
                        operator: gideonOperators.get(include) || FilterOperator.UNKONWN,
                    });

                    isKey = true;
                    isValue = false;
                    isLiteralValue = false;
                    isLiteralValueClosed = false;

                    key = "";
                    include = "";
                    value = "";
                    values = [];
                } else {
                    out.errors = ["Search query is incorrect. ", "Keys are required and cannot be empty. "];
                    return out;
                }
                // delimeter between values
            } else if (query[i] === ",") {
                key = key.trim();

                if (key) {
                    if (key.includes(" ") || key.includes("\n")) {
                        out.errors = ["Search query is incorrect. ", `Key error (${key}). `];
                        return out;
                    }

                    if (isLiteralValue || (!isLiteralValue && value.trim())) {
                        if (isLiteralValue && !isLiteralValueClosed) {
                            out.errors = [
                                "Search query is incorrect. ",
                                `Values error (${value}). `,
                                "Commas or double quotes are required. ",
                            ];
                            return out;
                        } else if (!isLiteralValue && (value.trim().includes(" ") || value.trim().includes("\n"))) {
                            out.errors = [
                                "Search query is incorrect. ",
                                `Values error (${value}). `,
                                "Commas or double quotes are required. ",
                            ];
                            return out;
                        }

                        values.push(isLiteralValue ? value : value.trim());
                    } else {
                        out.errors = ["Search query is incorrect. ", "Value error. "];
                        return out;
                    }
                } else {
                    out.errors = ["Search query is incorrect. ", "Keys are required and cannot be empty. "];
                    return out;
                }
                value = "";
                isLiteralValue = false;
                isLiteralValueClosed = false;
                // value continuation
            } else if (isValue) {
                if (isLiteralValue && isLiteralValueClosed) {
                    out.errors = [
                        "Search query is incorrect. ",
                        `Values error ("${value}" ${query[i]}). `,
                        "Commas or double quotes are required. ",
                    ];
                    return out;
                }

                value += query[i];
                // key continuation
            } else if (isKey) {
                key += query[i];
                // no key, no value - then start of a non literal value
            } else if (query[i] !== " " && query[i] !== "\n") {
                if (isLiteralValue && isLiteralValueClosed) {
                    out.errors = [
                        "Search query is incorrect. ",
                        `Values error ("${value}" ${query[i]}). `,
                        "Commas or double quotes are required. ",
                    ];
                    return out;
                }

                isValue = true;
                isLiteralValue = false;
                isLiteralValueClosed = false;
                value += query[i];
            }
        }

        key = key.trim();

        if (key) {
            if (key.includes(" ") || key.includes("\n")) {
                out.errors = ["Search query is incorrect. ", `Key error (${key}). `];
                return out;
            }

            if (isLiteralValue || (!isLiteralValue && value.trim())) {
                if (isLiteralValue && !isLiteralValueClosed) {
                    out.errors = [
                        "Search query is incorrect. ",
                        `Values error (${value}). `,
                        "Commas or double quotes are required. ",
                    ];
                    return out;
                } else if (!isLiteralValue && (value.trim().includes(" ") || value.trim().includes("\n"))) {
                    out.errors = [
                        "Search query is incorrect. ",
                        `Values error (${value}). `,
                        "Commas or double quotes are required. ",
                    ];
                    return out;
                }

                values.push(isLiteralValue ? value : value.trim());
            } else {
                out.errors = ["Search query is incorrect. ", `No value (${key}). `];
                return out;
            }

            if (!gideonOperators.has(include)) {
                out.errors = ["Search query is incorrect. ", `Syntax error: "${include}". `];
                return out;
            }

            out.filter.push({
                key: key,
                values,
                operator: gideonOperators.get(include) || FilterOperator.UNKONWN,
            });
        } else if (value.trim() || include) {
            out.errors = ["Search query is incorrect. ", "Keys are required and cannot be empty. "];
            return out;
        }
    }

    return out;
};
