export enum FilterOperator {
    UNKONWN = "??!!",
    EQ = "==",
    NE = "!=",
    EL = "~=",
    NL = "!~=",
    GT = ">=",
    LT = "<=",
}

export type FilterField = {
    key: string;
    values: string[];
    operator: FilterOperator;
};

export type Filter = FilterField[];

export enum EventKind {
    Unspecified = 0,
    ProcExec    = 1,
    ExecveAt    = 3,
    PTrace      = 4,
    Connect     = 5,
    OpenAt      = 7,
    SSHSession  = 6,
}

export enum AddressFamily {
    Unspecified = 0,
    Inet        = 2,
    Itet6       = 10,
}

export enum SSHSessionKind {
    Unspecified = 0,
    SSH         = 1,
    Portoshell  = 2,
    YtJobshell  = 3,
}

type BaseEvent = {
    ts: number;
    time: Date;
    source: string;
    host: string;
    kind: EventKind;
    pid: number;
    name: string;
    ppid: number;
    parentName: string;
    uid: number;
    sessionID: number;
    container: string;
    podID: string;
    podSetID: string;
    nannyServiceID: string;
};

export type EventProcExec = BaseEvent & {
    exe: string;
    args: string[];
};

export type EventConnect = BaseEvent & {
    family: AddressFamily;
    dstAddr: string;
    dstPort: number;
};

export type EventPtrace = BaseEvent & {
    retCode: number;
    request: number;
    targetPID: number;
};

export type EventSSHSession = BaseEvent & {
    sshID: number;
    sshUser: string;
    sshTTY: string;
    sshKind: SSHSessionKind;
};

export type EventOpenAt = BaseEvent & {
    retCode: number;
    fd: number;
    filename: string;
    flags: number;
};

export type Event = EventProcExec | EventConnect | EventPtrace | EventSSHSession | EventOpenAt;

export const DecodeEvents = (objs: any[]): Event[] =>
    objs.map(
        (o: any): Event => {
            const baseEvent: BaseEvent = {
                ts: o.ts,
                time: new Date(o.ts / 1000000),
                source: o.source || "n/a",
                host: o.host,
                kind: o.kind,
                pid: o.proc.pid,
                name: o.proc.name,
                ppid: o.proc.ppid,
                parentName: o.proc.parent_name,
                uid: o.proc.uid,
                sessionID: o.proc.session_id,
                container: o.proc.container,
                podID: o.proc.pod_id,
                podSetID: o.proc.podset_id,
                nannyServiceID: o.proc.nanny_service_id,
            };

            switch (o.kind) {
                case EventKind.ProcExec:
                    return {
                        exe: o.proc_exec.exe,
                        args: o.proc_exec.args,
                        ...baseEvent,
                    };
                case EventKind.Connect:
                    return {
                        family: o.connect.family,
                        dstAddr: o.connect.dst_addr,
                        dstPort: o.connect.dst_port,
                        ...baseEvent,
                    };
                case EventKind.PTrace:
                    return {
                        retCode: o.ptrace.ret_code,
                        request: o.ptrace.request,
                        targetPID: o.ptrace.target,
                        ...baseEvent,
                    };
                case EventKind.SSHSession:
                    return {
                        sshID: o.ssh_session.id,
                        sshUser: o.ssh_session.user,
                        sshTTY: o.ssh_session.tty,
                        sshKind: o.ssh_session.kind,
                        ...baseEvent,
                    };
                case EventKind.OpenAt:
                    return {
                        retCode: o.open_at.ret_code,
                        fd: o.open_at.fd,
                        filename: o.open_at.filename,
                        flags: o.open_at.flags,
                        ...baseEvent,
                    }
                default:
                    throw new Error(`unsupported event kind: ${o.kind}`);
            }
        }
    );
