import PropTypes from "prop-types";
import React, {Component} from "react";
import {withRouter} from "react-router";

function isModifiedEvent(event) {
    return Boolean(event.metaKey || event.altKey || event.ctrlKey || event.shiftKey);
}

export default function makeRouteLink({prop = "href"} = {}) {
    return function (WrappedComponent) {
        const wrappedComponentName = WrappedComponent.displayName || WrappedComponent.name || "Component";

        return withRouter(
            class extends Component {
                static displayName = `routeLink(${wrappedComponentName})`;

                static propTypes = {
                    target: PropTypes.string,
                    replace: PropTypes.bool,
                    to: PropTypes.oneOfType([PropTypes.string, PropTypes.object]),
                    onClick: PropTypes.func,

                    match: PropTypes.object.isRequired,
                    location: PropTypes.object.isRequired,
                    history: PropTypes.object.isRequired,
                };

                getLinkLocation() {
                    const {to} = this.props;

                    if (typeof to === "string") {
                        return {pathname: to};
                    }

                    return to;
                }

                isSameLocationAsPage() {
                    const {location} = this.props;
                    const linkLocation = this.getLinkLocation();

                    return ["pathname", "search", "hash"].every((key) => {
                        const linkValue = linkLocation[key] || "";
                        const pageValue = location[key];

                        return linkValue === pageValue;
                    });
                }

                handleClick = (event) => {
                    const {history} = this.props;
                    const {replace, to, target, onClick} = this.props;

                    if (onClick) {
                        onClick(event);
                    }

                    if (
                        !event.defaultPrevented && // onClick prevented default
                        event.button === 0 && // ignore right clicks
                        !target && // let browser handle "target=_blank" etc.
                        !isModifiedEvent(event) && // ignore clicks with modifier keys
                        to // don't handle Links without prop "to"
                    ) {
                        event.preventDefault();

                        if (this.isSameLocationAsPage()) {
                            return;
                        }

                        if (replace) {
                            history.replace(to);
                        } else {
                            history.push(to);
                        }
                    }
                };

                render() {
                    const {replace, to, location, history, match, ...proxyProps} = this.props; // eslint-disable-line no-unused-vars
                    let href;

                    if (to) {
                        href = history.createHref(this.getLinkLocation());
                    }

                    const finalProps = {
                        ...proxyProps,
                        [prop]: href,
                        onClick: this.handleClick,
                    };

                    return <WrappedComponent {...finalProps} />;
                }
            }
        );
    };
}
