import React, {useState, useEffect, PropsWithChildren} from "react";
import {Loader} from "@yandex-infracloud-ui/common";
import {useDismounted} from "@yandex-infracloud-ui/libs";
import {map, finalize, takeUntil} from "rxjs/operators";

import {SessionView} from "../components/SessionView/SessionView";
import {useTitle} from "../utils/hooks";
import {Page} from "../components/layout/Page/Page";
import {useGideonApi} from "../services";
import {Event} from "../models/gideon";

import classes from "./session.module.css";

interface Props {
    isSaved: boolean;
}

export const SessionPage = ({isSaved}: PropsWithChildren<Props>) => {
    useTitle("Session");

    const gideonApi = useGideonApi();
    const dismounted = useDismounted();

    const [isLoading, setIsLoading] = useState(false);
    const [inProgress, setInProgress] = useState(false);
    const [loadError, setLoadError] = useState("");
    const [unixMode, setUnixMode] = useState<boolean>(false);
    const [events, setEvents] = useState<Event[]>([]);
    const parsedHash = new URLSearchParams(window.location.hash.substr(1));
    const key = parsedHash.get("key");
    const useUnixMode = parsedHash.get("unixMode") === "yes";

    useEffect(() => {
        if (!key) {
            return;
        }

        setLoadError("");
        setIsLoading(true);
        setInProgress(false);
        let eventsPromise = (isSaved ? gideonApi.fetchSavedSessionEvents(key) : gideonApi.fetchSessionEvents(key));

        eventsPromise
            .pipe(
                map((events) => setEvents(events)),
                finalize(() => setIsLoading(false)),
                takeUntil(dismounted)
            )
            .subscribe({
                error: (err: Error) => {
                    // TODO(buglloc): ugly
                    if (err.message === "export in progress") {
                        setInProgress(true);
                        return;
                    }

                    setLoadError(err.message)
                },
            });
    }, [dismounted, gideonApi, key, isSaved]);

    useEffect(() => {
        setUnixMode(useUnixMode);
    }, [dismounted, useUnixMode]);

    return (
        <Page>
            <div className={classes["events_view"]}>
                <div className={classes["events-body"]}>
                    {
                        (loadError && <span className={classes["events-error"]}>Ooops: {loadError}</span>) ||
                        (inProgress && <span>The session is being exported, it will take up to 10 minutes. Please try again later.</span>) ||
                        (isLoading && <div className={classes["loader-container"]}><Loader size="l" /></div>) ||
                        (unixMode ? <div></div> : <SessionView events={events} />)}
                </div>
            </div>
        </Page>
    );
};
