import {YandexCookies} from "@yandex-infracloud-ui/libs";
import React, {useContext} from "react";

import {ApiServiceName, User} from "../models";
import {Storage} from "./Storage";

const storage = new Storage();

export class Config {
    public static getReduxLoggerConfig() {
        return {
            collapsed: storage.getItem("reduxLogger.collapsed", true),
            diff: storage.getItem("reduxLogger.diff", false),
            duration: storage.getItem("reduxLogger.duration", false),
            enabled: storage.getItem("reduxLogger.enabled", true),
            logErrors: storage.getItem("reduxLogger.logErrors", false),
        };
    }
    private readonly _config: YandexConfig;
    private readonly _now: Date | null;
    private _services: Map<ApiServiceName, string>;
    private readonly _version: string;
    private yandexCookies = new YandexCookies();

    public get now(): Date | null {
        return this._now;
    }

    public get version(): string {
        return this._version;
    }

    public get rawConfig() {
        return this._config;
    }

    constructor(YC: any) {
        this._config = {
            ...YC,
            user: {
                login: this.yandexCookies.getYandexLogin(),
                uid: this.yandexCookies.getYandexUID(),
            },
        };

        this._now = this._config.now ? new Date(this._config.now) : null;
        this._version = this._config.version ?? "unknown";

        this._services = new Map(this._config.envConfig.services.map((r) => [r.name as ApiServiceName, r.endpoint]));
    }

    public getUser(): User {
        return this._config.user;
    }

    public getApiEndpointForService(service: ApiServiceName) {
        if (!this._services.has(service)) {
            throw new Error(`Unsupported service "${service}". Add it to environment config!`);
        }

        return this._services.get(service)!;
    }
}

export const ConfigContext = React.createContext<Config | null>(null);

export function useConfig(tolerant = false): Config | null {
    const config = useContext(ConfigContext);
    if (!tolerant && !config) {
        throw new Error("Expected value in ConfigContext");
    }

    return config;
}
