import {ApiCache, BaseApi, HttpMethod, IApiError} from "@yandex-infracloud-ui/libs";
import {Observable} from "rxjs";
import {map} from "rxjs/operators";

import {Filter, Event, DecodeEvents, FilterOperator} from "../../models/gideon";

type EventFilter = {
    key: string;
    values: string[];
    operator: FilterOperator;
};

type EventsHTTPRequest = {
    filter: EventFilter[];
};

type EventsHTTPResponse = {
    events?: string[];
    error?: string;
};

type SessionEventsHTTPRequest = {
    key: string;
};

type SuggestRequest = {
    fullSearch: boolean;
    key: string;
    filter: Filter;
    valuePrefix: string;
};

type SuggestHTTPRequest = {
    fullSearch: boolean;
    key: string;
    filter: EventFilter[];
    valuePrefix: string;
};

function parseSessionEvents(data: string) : Event[] {
    let events = JSON.parse(`[${data.trim().split(/\n/).join(',')}]`);
    // TODO(buglloc): ugly
    if (events.length === 1 && !!events[0].error) {
        throw Error(events[0].error);
    }

    return DecodeEvents(events || []);
}

export class GideonApi extends BaseApi {
    private _cache = new ApiCache();

    constructor(gideonHost: string) {
        super(`${gideonHost}api/`);
    }

    public fetchEvents(filters: Filter): Observable<Event[]> {
        const body: EventsHTTPRequest = {
            filter: filters,
        };

        return this.request<any, EventsHTTPRequest, EventsHTTPResponse>(HttpMethod.POST, "query", undefined, body).pipe(
            map((resp) => {
                if (resp.error) {
                    throw Error(resp.error);
                }

                return DecodeEvents(resp.events || []);
            })
        );
    }

    public fetchSessionEvents(key: string): Observable<Event[]> {
        const body: SessionEventsHTTPRequest = {
            key: key,
        };

        return this.request<any, SessionEventsHTTPRequest, EventsHTTPResponse>(
            HttpMethod.POST,
            "session",
            undefined,
            body,
            false
        ).pipe(
            map((resp) => parseSessionEvents(String(resp)))
        );
    }

    public fetchSavedSessionEvents(key: string): Observable<Event[]> {
        const body: SessionEventsHTTPRequest = {
            key: key,
        };

        return this.request<any, SessionEventsHTTPRequest, EventsHTTPResponse>(
            HttpMethod.POST,
            "saved-session",
            undefined,
            body,
            false
        ).pipe(
            map((resp) => parseSessionEvents(String(resp)))
        );
    }

    public suggestValue(req: SuggestRequest): Observable<string[]> {
        return this.request<any, SuggestHTTPRequest, any>(HttpMethod.POST, "suggest", undefined, req).pipe(
            map((resp) => resp.values || [])
        );
    }

    protected getCSRF(): string {
        return "";
    }

    protected getFetchParams(method: HttpMethod, body: string | undefined, headers: HeadersInit): RequestInit {
        return {
            body,
            headers,
            method,
        };
    }

    protected handleError(resp: Response, error: IApiError): void {}
}
