package config

import (
	"context"
	"fmt"
	"net/url"
	"os"
	"strings"
	"time"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/file"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/tvm"
)

const (
	WebShutdownDeadline    = 10 * time.Second
	ExportShutdownDeadline = 10 * time.Minute
)

type ClickHouse struct {
	Addrs     []string      `yaml:"addrs"`
	Database  string        `yaml:"database"`
	User      string        `yaml:"user"`
	Password  string        `yaml:"password"`
	Compress  bool          `yaml:"compress"`
	Debug     bool          `yaml:"debug"`
	BatchSize int           `yaml:"batch_size"`
	BatchTick time.Duration `yaml:"batch_tick"`
}

type S3 struct {
	Endpoint        string `yaml:"endpoint"`
	Bucket          string `yaml:"bucket"`
	AccessKeyID     string `yaml:"access_key_id"`
	AccessSecretKey string `yaml:"access_secret_key"`
}

type YT struct {
	Proxy string `yaml:"proxy"`
	Path  string `yaml:"path"`
	Token string `yaml:"token"`
}

type Exporter struct {
	InSyncDays int           `yaml:"in_sync_days"`
	CheckTick  time.Duration `yaml:"check_tick"`
}

type ACL struct {
	StaffUsers  []string       `yaml:"staff_users"`
	StaffGroups []string       `yaml:"staff_groups"`
	AbcRoles    []string       `yaml:"abc_roles"`
	TVM         []tvm.ClientID `yaml:"tvm_ids"`
}

type Masker struct {
	Enabled  bool   `yaml:"enabled"`
	Upstream string `yaml:"upstream"`
}

type Config struct {
	Addr       string     `yaml:"addr"`
	Dev        bool       `yaml:"dev"`
	FrontPath  string     `yaml:"front_path"`
	LogLevel   string     `yaml:"log_level"`
	SessionKey string     `yaml:"session_key"`
	Masker     Masker     `yaml:"masker"`
	ClickHouse ClickHouse `yaml:"clickhouse"`
	S3         S3         `yaml:"s3"`
	YT         YT         `yaml:"yt"`
	Exporter   Exporter   `yaml:"exporter"`
	ACL        ACL        `yaml:"acl"`
}

func NewConfig(configPath string) (*Config, error) {
	cfg := &Config{
		LogLevel:  log.InfoString,
		Addr:      ":9090",
		FrontPath: "frontend",
		ACL: ACL{
			StaffUsers: []string{
				"buglloc",
			},
			StaffGroups: []string{
				"yandex_mnt_security_1327",
			},
		},
		Masker: Masker{
			Enabled:  false,
			Upstream: "http://[::1]:10203",
		},
		SessionKey: os.Getenv("SESSION_SIGN_KEY"),
		S3: S3{
			Endpoint:        "s3.mdst.yandex.net",
			Bucket:          "gsessions",
			AccessKeyID:     "Y6UGRgZiUgtec2Kna1Kg",
			AccessSecretKey: os.Getenv("S3_SECRET_KEY"),
		},
		YT: YT{
			Proxy: "locke",
			Path:  "//home/security/gideon-viewer/dev",
			Token: os.Getenv("YT_TOKEN"),
		},
		Exporter: Exporter{
			InSyncDays: 7,
			CheckTick:  1 * time.Hour,
		},
		ClickHouse: ClickHouse{
			Database:  "gideon",
			Password:  os.Getenv("CH_PASSWD"),
			Compress:  true,
			BatchSize: 1024,
			BatchTick: 100 * time.Millisecond,
		},
	}

	loader := confita.NewLoader(file.NewBackend(configPath))
	if err := loader.Load(context.Background(), cfg); err != nil {
		return nil, fmt.Errorf("failed to parse config: %w", err)
	}

	return cfg, nil
}

func (c *ClickHouse) URI(opts url.Values) string {
	if opts == nil {
		opts = url.Values{}
	}

	opts.Set("database", c.Database)
	opts.Set("username", c.User)
	opts.Set("password", c.Password)
	if c.Compress {
		opts.Set("compress", "true")
	}
	if c.Debug {
		opts.Set("debug", "true")
	}
	if len(c.Addrs) > 1 {
		opts.Set("alt_hosts", strings.Join(c.Addrs[1:], ","))
	}
	uri := fmt.Sprintf("tcp://%s?%s", c.Addrs[0], opts.Encode())

	if len(c.Addrs) > 1 {
		uri = fmt.Sprintf("%s&alt_hosts=%s", uri, strings.Join(c.Addrs[1:], ","))
	}
	return uri
}
