package filter

import (
	"fmt"
	"strconv"
	"strings"

	"a.yandex-team.ru/security/gideon/gideon/pkg/events"
	"a.yandex-team.ru/security/gideon/viewer/internal/anytime"
)

//go:generate easyjson

const (
	OperatorEQ = "=="
	OperatorNE = "!="
	OperatorEL = "~="
	OperatorNL = "!~="
	OperatorGT = ">="
	OperatorLT = "<="

	KeyTS             = "TS"
	KeyKind           = "Kind"
	KeySource         = "Source"
	KeyHost           = "Host"
	KeyPodID          = "Proc_PodID"
	KeyPodSetID       = "Proc_PodSetID"
	KeyNannyID        = "Proc_NannyID"
	KeySessionID      = "Proc_SessionID"
	KeySSHSessionID   = "SSHSession_ID"
	KeySSHSessionKind = "SSHSession_Kind"
)

//easyjson:json
type Expression struct {
	Key      string   `json:"key"`
	Values   []string `json:"values"`
	Operator string   `json:"operator"`
}

//easyjson:json
type Filter []Expression

func BuildWhere(filter Filter) (string, []interface{}, error) {
	var (
		where strings.Builder
		args  []interface{}
	)

	for i, e := range filter {
		if len(e.Values) == 0 {
			continue
		}

		key := DBKey(e.Key)
		if key == "" {
			return "", nil, fmt.Errorf("unexpected filter key: %s", e.Key)
		}

		op, ok := parseOperator(key, e.Operator)
		if !ok {
			return "", nil, fmt.Errorf("invalid operator (key=%q): %s", e.Key, e.Operator)
		}

		if i > 0 {
			where.WriteString(" AND ")
		}
		i++

		values, err := filterValues(key, e.Values)
		if err != nil {
			return "", nil, fmt.Errorf("value error (key=%q): %w", e.Key, err)
		}

		args = append(args, values...)
		switch {
		case len(e.Values) == 1:
			where.WriteString(key)
			where.WriteString(op)
			where.WriteByte('?')
		default:
			where.WriteByte('(')
			for idx := 0; idx < len(e.Values); idx++ {
				if idx > 0 {
					where.WriteString(" OR ")
				}

				where.WriteString(key)
				where.WriteString(op)
				where.WriteByte('?')
			}
			where.WriteByte(')')
		}
	}

	return where.String(), args, nil
}

func filterValues(key string, values []string) ([]interface{}, error) {
	out := make([]interface{}, len(values))
	for i, v := range values {
		switch key {
		case KeyKind:
			switch v {
			case "ProcExec":
				out[i] = events.EventKind_EK_PROC_EXEC
			case "ExecveAt":
				out[i] = events.EventKind_EK_SYS_EXECVE_AT
			case "Ptrace":
				out[i] = events.EventKind_EK_SYS_PTRACE
			case "Connect":
				out[i] = events.EventKind_EK_SYS_CONNECT
			case "OpenAt":
				out[i] = events.EventKind_EK_SYS_OPEN_AT
			case "NewSession":
				out[i] = events.EventKind_EK_SSH_SESSION
			default:
				return nil, fmt.Errorf("unsupported event kind: %s", v)
			}

		case KeySessionID:
			val, err := strconv.ParseUint(v, 10, 32)
			if err != nil {
				return nil, fmt.Errorf("invalid session_id: %w", err)
			}

			out[i] = uint32(val)

		case KeyTS:
			val, err := anytime.Parse(v)
			if err != nil {
				return nil, fmt.Errorf("invalid timestamp: %w", err)
			}

			out[i] = val

		case KeySSHSessionKind:
			switch v {
			case "ssh":
				out[i] = events.SessionKind_SK_SSH
			case "portoshell":
				out[i] = events.SessionKind_SK_PORTOSHELL
			case "yt_jobshell":
				out[i] = events.SessionKind_SK_YT_JOBSHELL
			default:
				return nil, fmt.Errorf("unsupported session kind: %s", v)
			}

		default:
			out[i] = v
		}
	}
	return out, nil
}

func DBKey(key string) string {
	switch key {
	case "kind":
		return KeyKind
	case "source":
		return KeySource
	case "host":
		return KeyHost
	case "pod_id":
		return KeyPodID
	case "pod_set_id":
		return KeyPodSetID
	case "session_id":
		return KeySessionID
	case "ssh_session_id":
		return KeySSHSessionID
	case "ssh_session_kind":
		return KeySSHSessionKind
	case "nanny_service_id":
		return KeyNannyID
	case "time":
		return KeyTS
	default:
		return ""
	}
}

func parseOperator(key, condition string) (string, bool) {
	switch key {
	case KeyTS:
		switch condition {
		case OperatorGT:
			return ">=", true
		case OperatorLT:
			return "<=", true
		}

		return "", false
	default:
		switch condition {
		case OperatorEQ:
			return "==", true
		case OperatorNE:
			return "!=", true
		case OperatorEL:
			return " LIKE ", true
		case OperatorNL:
			return " NOT LIKE ", true
		}

		return "", false
	}
}
