package export

import (
	"context"
	"errors"
	"fmt"
	"time"

	"github.com/cenkalti/backoff/v4"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/gideon/viewer/internal/config"
	"a.yandex-team.ru/security/gideon/viewer/internal/db"
	"a.yandex-team.ru/security/gideon/viewer/internal/sessionstorage"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
	"a.yandex-team.ru/yt/go/yterrors"
	"a.yandex-team.ru/yt/go/ytlock"
)

const (
	backoffTimeout = 10 * time.Second
)

type App struct {
	db             *db.DB
	log            log.Logger
	sessionStorage *sessionstorage.Storage
	yt             yt.Client
	cfg            *config.Config
	mainCtx        context.Context
	shutdownFn     context.CancelFunc
	done           chan struct{}
}

func NewApp(cfg *config.Config, logger log.Logger) (*App, error) {
	gideonDB, err := db.NewDB(cfg.ClickHouse)
	if err != nil {
		return nil, fmt.Errorf("failed to create DB: %w", err)
	}

	sessStorage, err := sessionstorage.NewStorage(sessionstorage.Options{
		Bucket:          cfg.S3.Bucket,
		Endpoint:        cfg.S3.Endpoint,
		AccessKeyID:     cfg.S3.AccessKeyID,
		SecretAccessKey: cfg.S3.AccessSecretKey,
	}, logger)
	if err != nil {
		return nil, fmt.Errorf("failed to create session storage: %w", err)
	}

	ytClient, err := ythttp.NewClient(&yt.Config{
		Proxy: cfg.YT.Proxy,
		Token: cfg.YT.Token,
	})
	if err != nil {
		return nil, fmt.Errorf("failed to create YT client: %w", err)
	}

	ctx, cancelFn := context.WithCancel(context.Background())
	return &App{
		db:             gideonDB,
		log:            logger,
		sessionStorage: sessStorage,
		yt:             ytClient,
		cfg:            cfg,
		mainCtx:        ctx,
		shutdownFn:     cancelFn,
		done:           make(chan struct{}),
	}, nil
}

func (a *App) Start() error {
	defer close(a.done)

	lock := ytlock.NewLockOptions(a.yt, ypath.Path(a.cfg.YT.Path).JoinChild("export-lock"), ytlock.Options{
		CreateIfMissing: true,
		LockMode:        yt.LockExclusive,
	})

	err := backoff.RetryNotify(
		func() error {
			ctx := context.Background()
			tx, err := lock.Acquire(ctx)
			if err != nil {
				if ctx.Err() != nil {
					return backoff.Permanent(err)
				}

				return err
			}
			defer func() { _ = lock.Release(ctx) }()

			err = a.export(ctx, tx)
			if err != nil {
				if ctx.Err() != nil {
					return backoff.Permanent(err)
				}
				return err
			}

			return nil
		},
		backoff.WithContext(backoff.NewConstantBackOff(backoffTimeout), a.mainCtx),
		func(err error, sleep time.Duration) {
			if a.mainCtx.Err() != nil {
				// that's fine
				return
			}

			if yterrors.ContainsErrorCode(err, yterrors.CodeConcurrentTransactionLockConflict) {
				// that's fine
				return
			}

			a.log.Error("export failed", log.Duration("sleep", sleep), log.Error(err))
		},
	)

	if err != nil && a.mainCtx.Err() == nil {
		return fmt.Errorf("failed to watch stages")
	}

	a.log.Info("watcher stopped")
	return nil
}

func (a *App) export(ctx context.Context, lostTx <-chan struct{}) error {
	meta, err := a.restoreMeta()
	if err != nil {
		return fmt.Errorf("failed to restore meta: %w", err)
	}

	for {
		select {
		case <-a.mainCtx.Done():
			// that's fine
			a.log.Info("app context canceled")
			return nil
		case <-ctx.Done():
			return ctx.Err()
		case <-lostTx:
			return errors.New("lost YT lock")
		case <-time.After(a.cfg.Exporter.CheckTick):
			// ok, let's export something!
		}

		syncTo := time.Now().Add(-1 * time.Minute).UnixNano()
		if err := a.processExistedSessions(ctx, meta.LastSync, syncTo); err != nil {
			a.log.Error("can't process existed sessions", log.Error(err))
		}

		if err := a.processNewSessions(ctx, meta.LastSync, syncTo); err != nil {
			a.log.Error("can't process new sessions", log.Error(err))
		}

		meta.LastSync = syncTo
		if err := a.saveMeta(meta); err != nil {
			a.log.Error("failed to save meta", log.Error(err))
		}
	}
}

func (a *App) Shutdown(ctx context.Context) error {
	a.shutdownFn()

	select {
	case <-a.done:
		return nil
	case <-ctx.Done():
		return ctx.Err()
	}
}
