package export

import (
	"context"
	"fmt"

	"github.com/mailru/easyjson/jwriter"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/gideon/viewer/internal/models"
)

type sessionUpdate struct {
	info   models.SSHSessionInfo
	writer *jwriter.Writer
}

func (a *App) processExistedSessions(ctx context.Context, from, to int64) error {
	a.log.Info("process existed sessions")
	defer a.log.Info("process existed sessions: done")

	process := func(from, to int64, sessInfos []models.SSHSessionInfo) error {
		dbEvents, err := a.db.QuerySessionsEvents(ctx, from, to, sessInfos)
		if err != nil {
			return err
		}
		a.log.Info("fetched new events", log.Int("count", len(dbEvents)))

		toUpdate := make(map[models.SSHSessionInfo]sessionUpdate)
		for _, event := range dbEvents {
			i := models.SSHSessionInfo{
				Host:      event.Host,
				PodID:     event.Proc.PodID,
				SessionID: event.Proc.SessionID,
			}

			if _, ok := toUpdate[i]; !ok {
				found := false
				for _, sess := range sessInfos {
					if sess.Host == i.Host && sess.PodID == i.PodID && sess.SessionID == i.SessionID {
						toUpdate[i] = sessionUpdate{
							info:   sess,
							writer: &jwriter.Writer{},
						}
						found = true
						break
					}
				}

				if !found {
					a.log.Error("can't find session for event", log.Any("event", i))
					continue
				}
			}

			event.MarshalEasyJSON(toUpdate[i].writer)
			toUpdate[i].writer.RawByte('\n')
		}

		for _, sess := range toUpdate {
			data, err := sess.writer.BuildBytes()
			if err != nil {
				a.log.Error("can't marshal session events", log.Any("session", sess.info), log.Error(err))
				continue
			}

			oldSess, err := a.sessionStorage.DownloadSession(ctx, sess.info)
			if err != nil {
				a.log.Error("can't download old session events", log.Any("session", sess.info), log.Error(err))
				continue
			}

			err = a.sessionStorage.UploadSession(ctx, sess.info, append(oldSess, data...))
			if err != nil {
				a.log.Error("can't save updated session", log.Any("session", sess.info), log.Error(err))
				continue
			}
		}

		return nil
	}

	for i := 0; i < a.cfg.Exporter.InSyncDays; i++ {
		sessions, err := a.sessionStorage.ListSessions(ctx, i)
		if err != nil {
			return fmt.Errorf("can't list sessions: %w", err)
		}

		a.log.Info("got sessions", log.Int("count", len(sessions)))
		err = process(from, to, sessions)
		if err != nil {
			return fmt.Errorf("can't process sessions: %w", err)
		}
	}

	return nil
}

func (a *App) processNewSessions(ctx context.Context, from, to int64) error {
	a.log.Info("process new sessions")
	defer a.log.Info("process new sessions: done")

	process := func(from, to int64, sessInfo models.SSHSessionInfo) error {
		dbEvents, err := a.db.QuerySessionEvents(ctx, sessInfo)
		if err != nil {
			return err
		}

		writer := &jwriter.Writer{}
		for _, event := range dbEvents {
			event.MarshalEasyJSON(writer)
			writer.RawByte('\n')
		}

		data, err := writer.BuildBytes()
		if err != nil {
			return fmt.Errorf("can't marshal session events: %w", err)
		}

		err = a.sessionStorage.UploadSession(ctx, sessInfo, data)
		if err != nil {
			return fmt.Errorf("can't save updated session: %w", err)
		}

		err = a.sessionStorage.DeleteNewSession(ctx, sessInfo)
		if err != nil {
			return fmt.Errorf("can't delete new session: %w", err)
		}

		return nil
	}

	sessions, err := a.sessionStorage.ListNewSessions(ctx)
	if err != nil {
		return fmt.Errorf("can't list sessions: %w", err)
	}

	a.log.Info("got sessions", log.Int("count", len(sessions)))
	for _, sess := range sessions {
		err = process(from, to, sess)
		if err != nil {
			a.log.Error("can't process sessions", log.Any("session", sess), log.Error(err))
			continue
		}
	}

	return nil
}
