package www

import (
	"errors"
	"fmt"
	"net/http"
	"time"

	"github.com/mailru/easyjson"
	"github.com/mailru/easyjson/jwriter"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/gideon/viewer/internal/db"
	"a.yandex-team.ru/security/gideon/viewer/internal/filter"
	"a.yandex-team.ru/security/gideon/viewer/internal/models"
	"a.yandex-team.ru/security/gideon/viewer/internal/sessionstorage"
)

const (
	unixNano = uint64(time.Second)
	minTS    = uint64(1000000000000000000)
)

func (a *App) queryHandler(w http.ResponseWriter, r *http.Request) {
	var body queryRequest
	err := easyjson.UnmarshalFromReader(r.Body, &body)
	if err != nil {
		apiErrResponse(w, err)
		return
	}

	if len(body.Filter) == 0 {
		_, _, _ = easyjson.MarshalToHTTPResponseWriter(
			queryResponse{Ok: true, Events: []models.JsEvent{}},
			w,
		)
		return
	}

	where, args, err := filter.BuildWhere(body.Filter)
	if err != nil {
		apiErrResponse(w, err)
		return
	}

	dbEvents, err := a.db.QueryEvents(r.Context(), where, args)
	if err != nil {
		apiErrResponse(w, err)
		return
	}

	data, err := easyjson.Marshal(queryResponse{Ok: true, Events: dbEvents})
	if err != nil {
		apiErrResponse(w, err)
		return
	}

	if a.masker != nil {
		masked, err := a.masker.Mask(r.Context(), data)
		if err != nil {
			a.log.Error("failed to mask session content, use original", log.Error(err))
		} else {
			data = masked
		}
	}

	_, _ = w.Write(data)
}

func (a *App) suggestHandler(w http.ResponseWriter, r *http.Request) {
	var body suggestRequest
	err := easyjson.UnmarshalFromReader(r.Body, &body)
	if err != nil {
		apiErrResponse(w, err)
		return
	}

	key := filter.DBKey(body.Key)
	if key == "" || key == filter.KeyKind {
		apiErrResponse(w, fmt.Errorf("unsupported filter key: %s", body.Key))
	}

	where, args, err := filter.BuildWhere(body.Filter)
	if err != nil {
		apiErrResponse(w, err)
		return
	}

	dbCandidates, err := a.db.SuggestKey(r.Context(), db.SuggestRequest{
		Key:         key,
		ValuePrefix: body.ValuePrefix,
		FullSearch:  body.FullSearch,
		Args:        args,
		Where:       where,
	})
	if err != nil {
		apiErrResponse(w, err)
		return
	}

	_, _, _ = easyjson.MarshalToHTTPResponseWriter(
		suggestResponse{Ok: true, Values: dbCandidates},
		w,
	)
}

func (a *App) getSessionHandler(w http.ResponseWriter, r *http.Request) {
	var body gewSessionRequest
	err := easyjson.UnmarshalFromReader(r.Body, &body)
	if err != nil {
		apiErrResponse(w, err)
		return
	}

	if body.Key == "" {
		apiErrResponse(w, errors.New("no key specified"))
		return
	}

	var sessInfo models.SSHSessionInfo
	if authPassedFromContext(r.Context()) {
		sessInfo, err = a.sessionSigner.ParseLax(body.Key)
	} else {
		sessInfo, err = a.sessionSigner.Parse(body.Key)
	}

	if err != nil {
		apiErrResponse(w, err)
		return
	}

	dbEvents, err := a.db.QuerySessionEvents(r.Context(), sessInfo)
	if err != nil {
		apiErrResponse(w, err)
		return
	}

	jw := &jwriter.Writer{}
	for _, e := range dbEvents {
		e.MarshalEasyJSON(jw)
		jw.RawByte('\n')
	}

	data, err := jw.BuildBytes()
	if err != nil {
		apiErrResponse(w, err)
		return
	}

	if a.masker != nil {
		masked, err := a.masker.Mask(r.Context(), data)
		if err != nil {
			a.log.Error("failed to mask session content, use original", log.Error(err))
		} else {
			data = masked
		}
	}

	w.Header().Set("Content-Type", "application/x-jsonlines")
	_, _ = w.Write(data)
}

func (a *App) getSavedSessionHandler(w http.ResponseWriter, r *http.Request) {
	var body getSavedSessionRequest
	err := easyjson.UnmarshalFromReader(r.Body, &body)
	if err != nil {
		apiErrResponse(w, err)
		return
	}

	if body.Key == "" {
		apiErrResponse(w, errors.New("no key specified"))
		return
	}

	var sessInfo models.SSHSessionInfo
	if authPassedFromContext(r.Context()) {
		sessInfo, err = a.sessionSigner.ParseLax(body.Key)
	} else {
		sessInfo, err = a.sessionSigner.Parse(body.Key)
	}

	if err != nil {
		apiErrResponse(w, err)
		return
	}

	data, err := a.sessionStorage.DownloadSession(r.Context(), sessInfo)
	if err != nil {
		if sessionstorage.IsErrCode(err, sessionstorage.ErrCodeNoSuchKey) && a.sessionStorage.IsNewSessionExist(r.Context(), sessInfo) {
			_, _, _ = easyjson.MarshalToHTTPResponseWriter(errResponse{Ok: false, Err: "export in progress"}, w)
			return
		}

		apiErrResponse(w, err)
		return
	}

	if a.masker != nil {
		masked, err := a.masker.Mask(r.Context(), data)
		if err != nil {
			a.log.Error("failed to mask session content, use original", log.Error(err))
		} else {
			data = masked
		}
	}

	w.Header().Set("Content-Type", "application/x-jsonlines")
	_, _ = w.Write(data)
}

func (a *App) exportSessionHandler(w http.ResponseWriter, r *http.Request) {
	var body newSessionRequest
	err := easyjson.UnmarshalFromReader(r.Body, &body)
	if err != nil {
		apiErrResponse(w, err)
		return
	}

	ts := body.TS
	if ts < minTS {
		ts *= unixNano
	}

	sessEvent, err := a.db.QuerySSHSession(r.Context(), db.QuerySSHSessionRequest{
		TS:           ts,
		SessionID:    body.SessionID,
		SSHSessionID: body.SSHSessionID,
	})
	if err != nil {
		apiErrResponse(w, err)
		return
	}

	sessData, err := sessEvent.MarshalJSON()
	if err != nil {
		apiErrResponse(w, err)
		return
	}

	sessInfo := models.SSHSessionInfoFromEvent(sessEvent)
	if !a.sessionStorage.IsSessionExist(r.Context(), sessInfo) {
		err = a.sessionStorage.NewSession(r.Context(), sessInfo, sessData)
		if err != nil {
			apiErrResponse(w, err)
			return
		}
	}

	key, err := a.sessionSigner.Sign(sessInfo)
	if err != nil {
		apiErrResponse(w, err)
		return
	}

	_, _, _ = easyjson.MarshalToHTTPResponseWriter(
		newSessionResponse{
			Ok:  true,
			Key: key,
			URL: fmt.Sprintf("https://%s/saved-session#key=%s", r.Host, key),
			Title: fmt.Sprintf(
				"host=%s pod=%s user=%s ssh_id=%s sess_id=%d",
				sessEvent.Host, sessEvent.Proc.PodID,
				sessEvent.SSHSession.User,
				body.SSHSessionID, sessEvent.Proc.SessionID,
			),
		},
		w,
	)
}

func apiErrResponse(w http.ResponseWriter, err error) {
	_, _, _ = easyjson.MarshalToHTTPResponseWriter(errResponse{Ok: false, Err: err.Error()}, w)
}
