package www

import (
	"context"
	"fmt"
	"net/http"
	"os"

	"github.com/go-chi/chi/v5"
	"github.com/go-chi/chi/v5/middleware"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/httputil/middleware/recovery"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/security/gideon/viewer/internal/config"
	"a.yandex-team.ru/security/gideon/viewer/internal/db"
	"a.yandex-team.ru/security/gideon/viewer/internal/masker"
	"a.yandex-team.ru/security/gideon/viewer/internal/sessionkey"
	"a.yandex-team.ru/security/gideon/viewer/internal/sessionstorage"
	"a.yandex-team.ru/security/libs/go/acler"
	"a.yandex-team.ru/security/libs/go/chim"
)

type App struct {
	db             *db.DB
	log            log.Logger
	https          http.Server
	sessionSigner  *sessionkey.Signer
	sessionStorage *sessionstorage.Storage
	isDev          bool
	masker         *masker.Masker
	acl            *acler.ACLer
	bbClient       *httpbb.Client
	tvmClient      *tvmtool.Client
}

func NewApp(cfg *config.Config, logger log.Logger) (*App, error) {
	gideonDB, err := db.NewDB(cfg.ClickHouse)
	if err != nil {
		return nil, fmt.Errorf("failed to create DB: %w", err)
	}

	sessStorage, err := sessionstorage.NewStorage(sessionstorage.Options{
		Bucket:          cfg.S3.Bucket,
		Endpoint:        cfg.S3.Endpoint,
		AccessKeyID:     cfg.S3.AccessKeyID,
		SecretAccessKey: cfg.S3.AccessSecretKey,
	}, logger)
	if err != nil {
		return nil, fmt.Errorf("failed to create session storage: %w", err)
	}

	var maskClient *masker.Masker
	if cfg.Masker.Enabled {
		maskClient = masker.NewMasker(cfg.Masker.Upstream)
	}

	app := &App{
		isDev:          cfg.Dev,
		db:             gideonDB,
		sessionSigner:  sessionkey.NewSigner([]byte(cfg.SessionKey)),
		sessionStorage: sessStorage,
		masker:         maskClient,
		log:            logger,
	}

	if !app.isDev {
		app.tvmClient, err = tvmtool.NewAnyClient(tvmtool.WithLogger(logger.Structured()))
		if err != nil {
			return nil, fmt.Errorf("failed to create TVMTool client: %w", err)
		}

		app.bbClient, err = httpbb.NewIntranet(httpbb.WithTVM(app.tvmClient), httpbb.WithLogger(logger.Structured()))
		if err != nil {
			return nil, fmt.Errorf("failed to create BB client: %w", err)
		}

		app.acl, err = acler.New(
			acler.User(cfg.ACL.StaffUsers...),
			acler.Department(cfg.ACL.StaffGroups...),
			acler.AbcRole(cfg.ACL.AbcRoles...),
			acler.TVM(cfg.ACL.TVM...),
		)
		if err != nil {
			return nil, fmt.Errorf("failed to create acler: %w", err)
		}
	}

	app.https = http.Server{
		Addr:    cfg.Addr,
		Handler: app.router(cfg),
	}

	return app, nil
}

func (a *App) Start() error {
	a.log.Infof("app starting at: %s", a.https.Addr)
	return a.https.ListenAndServe()
}

func (a *App) Shutdown(ctx context.Context) error {
	return a.https.Shutdown(ctx)
}

func (a *App) router(cfg *config.Config) http.Handler {
	r := chi.NewRouter()
	r.Use(recovery.New(recovery.WithLogger(a.log.Structured())))
	r.Use(NoFramesMiddleware)

	r.Get("/ping/liveness", func(w http.ResponseWriter, _ *http.Request) {
		_, _ = w.Write([]byte("OK"))
	})

	r.Get("/ping/readiness", func(w http.ResponseWriter, r *http.Request) {
		if err := a.db.Ping(r.Context()); err != nil {
			http.Error(w, fmt.Sprintf("db fail: %v", err), http.StatusBadRequest)
			return
		}

		_, _ = w.Write([]byte("OK"))
	})

	r.Route("/api", func(r chi.Router) {
		r.Use(chim.LogRequest(a.log))
		if a.isDev {
			r.Use(CORSMiddleware, FakeAuthMiddleware)
			r.Options("/*", func(writer http.ResponseWriter, request *http.Request) {})
		} else {
			r.Use(middleware.RealIP, AuthMiddleware(a.bbClient, a.tvmClient, a.acl))
		}

		r.Group(func(r chi.Router) {
			if !a.isDev {
				r.Use(CheckAuthMiddleware)
			}

			r.Post("/query", a.queryHandler)
			r.Post("/suggest", a.suggestHandler)
			r.Post("/sign-session", a.exportSessionHandler)
			r.Post("/export-session", a.exportSessionHandler)
		})

		r.Post("/session", a.getSessionHandler)
		r.Post("/saved-session", a.getSavedSessionHandler)
	})

	frontendFS := http.FileServer(http.Dir(cfg.FrontPath))
	r.Get("/*", func(w http.ResponseWriter, r *http.Request) {
		if _, err := os.Stat(cfg.FrontPath + r.URL.Path); os.IsNotExist(err) {
			w.Header().Set("Cache-Control", "public, max-age=10")
			http.StripPrefix(r.RequestURI, frontendFS).ServeHTTP(w, r)
		} else {
			frontendFS.ServeHTTP(w, r)
		}
	})

	return r
}
