package www

import (
	"context"
	"errors"
	"fmt"
	"net/http"

	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/security/libs/go/acler"
)

var errInvalidOAuthToken = errors.New("invalid OAuth token")
var errAccessDenied = errors.New("access denied")

type contextKey struct {
	key string
}

var contextAuthPassedKey = &contextKey{"auth-passed"}

func authPassedFromContext(ctx context.Context) bool {
	passed, ok := ctx.Value(contextAuthPassedKey).(bool)
	return ok && passed
}

func NoFramesMiddleware(next http.Handler) http.Handler {
	fn := func(w http.ResponseWriter, r *http.Request) {
		w.Header().Set("X-Frame-Options", "SAMEORIGIN")
		next.ServeHTTP(w, r)
	}
	return http.HandlerFunc(fn)
}

func CORSMiddleware(next http.Handler) http.Handler {
	fn := func(w http.ResponseWriter, r *http.Request) {
		w.Header().Set("Access-Control-Allow-Origin", "*")
		w.Header().Set("Access-Control-Allow-Headers", "*")
		w.Header().Set("Access-Control-Request-Method", "*")
		if r.Method == http.MethodOptions {
			return
		}

		next.ServeHTTP(w, r)
	}
	return http.HandlerFunc(fn)
}

func AuthMiddleware(bbClient blackbox.Client, tvmClient tvm.Client, acl *acler.ACLer) func(http.Handler) http.Handler {
	oauthAuth := func(r *http.Request) error {
		token := r.Header.Get("Authorization")
		if len(token) <= 6 || token[:6] != "OAuth " {
			return errInvalidOAuthToken
		}

		rsp, err := bbClient.OAuth(
			r.Context(),
			blackbox.OAuthRequest{
				OAuthToken: token[6:],
				UserIP:     r.RemoteAddr,
			})

		if err != nil {
			return fmt.Errorf("failed to check OAuth token: %w", err)
		}

		if !acl.CheckUser(rsp.User.Login) {
			return fmt.Errorf("user '%s' access denied", rsp.User.Login)
		}

		return nil
	}

	sessAuth := func(r *http.Request) error {
		sessionID, err := r.Cookie("Session_id")
		if err != nil {
			return err
		}

		rsp, err := bbClient.SessionID(r.Context(), blackbox.SessionIDRequest{
			SessionID: sessionID.Value,
			UserIP:    r.RemoteAddr,
			Host:      r.Host,
		})
		if err != nil {
			return fmt.Errorf("failed to auth in BB with sesion cookie: %w", err)
		}

		if !acl.CheckUser(rsp.User.Login) {
			return fmt.Errorf("user '%s' access denied", rsp.User.Login)
		}

		return nil
	}

	tvmAuth := func(r *http.Request) error {
		ticket := r.Header.Get("X-Ya-Service-Ticket")
		if ticket == "" {
			return errors.New("no 'Session_id' cookie")
		}

		ticketInfo, err := tvmClient.CheckServiceTicket(r.Context(), ticket)
		if err != nil {
			return err
		}

		if !acl.CheckTvm(ticketInfo.SrcID) {
			return fmt.Errorf("tvm app '%d' access denied", ticketInfo.SrcID)
		}

		return nil
	}

	return func(next http.Handler) http.Handler {
		fn := func(w http.ResponseWriter, r *http.Request) {
			err := oauthAuth(r)
			if err != nil && errors.Is(err, errInvalidOAuthToken) {
				err = sessAuth(r)
				if err != nil && errors.Is(err, http.ErrNoCookie) {
					err = tvmAuth(r)
				}
			}

			next.ServeHTTP(w, r.WithContext(context.WithValue(r.Context(), contextAuthPassedKey, err == nil)))
		}
		return http.HandlerFunc(fn)
	}
}

func FakeAuthMiddleware(next http.Handler) http.Handler {
	fn := func(w http.ResponseWriter, r *http.Request) {
		next.ServeHTTP(w, r.WithContext(context.WithValue(r.Context(), contextAuthPassedKey, true)))
	}
	return http.HandlerFunc(fn)
}

func CheckAuthMiddleware(next http.Handler) http.Handler {
	fn := func(w http.ResponseWriter, r *http.Request) {
		if !authPassedFromContext(r.Context()) {
			apiErrResponse(w, errAccessDenied)
			return
		}

		next.ServeHTTP(w, r)
	}
	return http.HandlerFunc(fn)
}
