package sessionkey

import (
	"crypto/hmac"
	"crypto/sha1"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"strings"

	"a.yandex-team.ru/security/gideon/viewer/internal/models"
)

type Signer struct {
	key []byte
}

func NewSigner(key []byte) *Signer {
	return &Signer{
		key: key,
	}
}

func (s *Signer) Parse(key string) (models.SSHSessionInfo, error) {
	var out models.SSHSessionInfo

	idx := strings.IndexByte(key, ':')
	if idx <= 0 || idx == len(key) {
		return out, ErrInvalidFormat
	}

	actualMAC, err := base64.URLEncoding.DecodeString(key[idx+1:])
	if err != nil {
		return out, fmt.Errorf("failed to parse sign: %w", err)
	}

	rawSessInfo, err := base64.URLEncoding.DecodeString(key[:idx])
	if err != nil {
		return out, fmt.Errorf("failed to parse sess body: %w", err)
	}

	mac := hmac.New(sha1.New, s.key)
	_, _ = mac.Write(rawSessInfo)
	expectedMAC := mac.Sum(nil)

	if !hmac.Equal(actualMAC, expectedMAC) {
		return out, ErrSignInvalid
	}

	err = json.Unmarshal(rawSessInfo, &out)
	return out, err
}

func (s *Signer) ParseLax(key string) (models.SSHSessionInfo, error) {
	var out models.SSHSessionInfo

	idx := strings.IndexByte(key, ':')
	if idx <= 0 {
		idx = len(key)
	}

	rawSessInfo, err := base64.URLEncoding.DecodeString(key[:idx])
	if err != nil {
		return out, fmt.Errorf("failed to parse sess body: %w", err)
	}

	err = json.Unmarshal(rawSessInfo, &out)
	return out, err
}

func (s *Signer) Sign(info models.SSHSessionInfo) (string, error) {
	jsonInfo, err := json.Marshal(info)
	if err != nil {
		return "", err
	}

	mac := hmac.New(sha1.New, s.key)
	_, _ = mac.Write(jsonInfo)
	sign := mac.Sum(nil)

	out := fmt.Sprintf(
		"%s:%s",
		base64.URLEncoding.EncodeToString(jsonInfo),
		base64.URLEncoding.EncodeToString(sign),
	)
	return out, nil
}
