package checker

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"os"
	"os/exec"
	"path/filepath"
	"strings"
	"time"

	"a.yandex-team.ru/security/hector/internal/config"
	"a.yandex-team.ru/security/hector/internal/procutils"
	"a.yandex-team.ru/security/hector/internal/remote"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

type ChildResults []struct {
	Path    string      `json:"path"`
	LineNo  int         `json:"line_no"`
	Problem interface{} `json:"problem"`
}

type HectorResult struct {
	URL     string      `json:"url"`
	LineNo  int         `json:"line_no"`
	Author  string      `json:"author"`
	Problem interface{} `json:"problem"`
}

type HectorOutput struct {
	Repo      remote.ExportedRepo `json:"repo"`
	LocalPath string              `json:"local_path"`
	Result    interface{}         `json:"result"`
}

func CheckRepo(repo remote.Repo) (completed bool) {
	startTime := time.Now()
	localPath := repo.GenLocalPath(config.WorkDir)
	defer cleanUp(localPath)

	skip, err := repo.Checkout(localPath, true)
	if err != nil {
		simplelog.Error("Failed to checkout repo", "url", repo.CloneURL(), "err", err.Error())
		if skip {
			completed = true
		}
		return
	}

	childResult, err := runChild(localPath)
	if err != nil {
		simplelog.Error("Failed to check repo", "path", localPath, "err", err.Error())
		return
	}

	if config.ParseStdout {
		err = writeEnrichmentResult(localPath, repo, childResult)
	} else {
		err = writeRawResult(localPath, repo, childResult)
	}

	if err != nil {
		simplelog.Error("Failed to check repo", "name", repo.Name(), "err", err.Error())
		return
	}

	elapsed := fmt.Sprintf("%.2fs", time.Since(startTime).Seconds())
	simplelog.Info("Complete repo: "+repo.Name(), "elapsed", elapsed)
	completed = true
	return
}

func runChild(repoPath string) (*procutils.CmdResult, error) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Duration(config.Timeout*1000000000))
	defer cancel() // The cancel should be deferred so resources are cleaned up

	args := append(config.ChildArgs, repoPath)
	cmd := exec.CommandContext(ctx, config.ChildBinary, args...)

	cwdInfo, err := os.Stat(repoPath)
	if err != nil {
		return nil, err
	}

	if !cwdInfo.IsDir() {
		cmd.Dir = filepath.Dir(repoPath)
	} else {
		cmd.Dir = repoPath
	}

	result, err := procutils.RunCmd(cmd)
	if ctx.Err() == context.DeadlineExceeded {
		return result, errors.New("child timed out")
	}

	if result != nil && result.ExitCode == 1 {
		command := fmt.Sprintf("%s %s", config.ChildBinary, strings.Join(args, " "))
		simplelog.Error("child command failed", "command", command, "stderr", result.Stderr)
	}

	return result, err
}

func writeRawResult(localPath string, repo remote.Repo, childResult *procutils.CmdResult) error {
	result := HectorOutput{
		Repo:      repo.Export(),
		LocalPath: localPath,
		Result:    childResult,
	}

	resultJSON, err := json.MarshalIndent(result, "", "  ")
	if err != nil {
		return err
	}

	if _, err := os.Stat(config.ResultDir); err != nil {
		if err = os.MkdirAll(config.ResultDir, 0777); err != nil {
			return err
		}
	}

	resultFile := repo.GenResultPath(config.ResultDir)
	err = ioutil.WriteFile(resultFile, resultJSON, 0644)
	if err != nil {
		return err
	}

	simplelog.Info("Result saved", "path", resultFile)
	return nil
}

func writeEnrichmentResult(localPath string, repo remote.Repo, checkResult *procutils.CmdResult) error {
	if checkResult.Stdout == "" {
		// Just skip it
		return nil
	}

	var rawResults ChildResults
	err := json.Unmarshal([]byte(checkResult.Stdout), &rawResults)
	if err != nil {
		return err
	}

	if len(rawResults) == 0 {
		// Just skip it
		return nil
	}

	results := make([]HectorResult, len(rawResults))
	for i, result := range rawResults {
		relativePath := strings.TrimPrefix(result.Path, localPath+"/")

		author, err := repo.Author(relativePath, result.LineNo)
		if err != nil {
			simplelog.Error("failed to get author", "path", result.Path, "line_no", result.LineNo, "err", err.Error())
			author = ""
		}

		results[i] = HectorResult{
			URL:     repo.PathToURL(relativePath, result.LineNo),
			LineNo:  result.LineNo,
			Author:  author,
			Problem: result.Problem,
		}
	}

	result := HectorOutput{
		Repo:      repo.Export(),
		LocalPath: localPath,
		Result:    results,
	}

	resultJSON, err := json.MarshalIndent(result, "", "  ")
	if err != nil {
		return err
	}

	if _, err := os.Stat(config.ResultDir); err != nil {
		if err = os.MkdirAll(config.ResultDir, 0777); err != nil {
			return err
		}
	}

	resultFile := repo.GenResultPath(config.ResultDir)
	err = ioutil.WriteFile(resultFile, resultJSON, 0644)
	if err != nil {
		return err
	}

	simplelog.Info("Enrichment result saved", "count", len(result.Result.([]HectorResult)), "path", resultFile)
	return nil
}

func cleanUp(localPath string) {
	if err := os.RemoveAll(localPath); err != nil {
		simplelog.Error("failed to cleanup", "path", localPath, "err", err.Error())
	}
}
