package commands

import (
	"fmt"
	"net/url"
	"strings"
	"sync"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/hector/internal/checker"
	"a.yandex-team.ru/security/hector/internal/cli"
	"a.yandex-team.ru/security/hector/internal/config"
	"a.yandex-team.ru/security/hector/internal/gitutils"
	"a.yandex-team.ru/security/hector/internal/remote/bbrepo"
	"a.yandex-team.ru/security/hector/internal/stash"
	"a.yandex-team.ru/security/hector/internal/state"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

var bbCmd = &cobra.Command{
	Use:   "bb [flags] -- /check-binary [child-flags]",
	Short: "Walk though bb.y-t.ru repos",
	RunE:  cli.WrapCobraCommand("bb", runBBCmd),
}

func init() {
	RootCmd.AddCommand(bbCmd)
}

func runBBCmd(oldState *state.RepoState, newState *state.RepoState) (err error) {
	gitErr := gitutils.CheckVersion()
	if gitErr != nil {
		simplelog.Warn(gitErr.Error())
	}

	if config.DryRun {
		fmt.Println("Acceptable repos: ")
	}

	stashURL, _ := url.Parse(config.BBBaseURL)
	stashClient := stash.NewClient(config.BBLogin, config.BBPassword, stashURL)

	jobs := make(chan stash.Repository, config.Concurrency)
	var wg sync.WaitGroup
	wg.Add(config.Concurrency + 1)
	for w := 0; w < config.Concurrency+1; w++ {
		go bbRepoWorker(stashClient, jobs, oldState, newState, &wg)
	}

	total := 0
	cb := func(repo stash.Repository) bool {
		if config.MaxRepos > 0 && total >= config.MaxRepos {
			// stop iterate
			return false
		}

		total += 1
		jobs <- repo
		return true
	}

	if len(config.Repos) > 0 {
		for _, repo := range config.Repos {
			splitted := strings.SplitN(repo, "/", 2)
			repo, err := stashClient.GetRepository(splitted[0], splitted[1])
			if err != nil {
				simplelog.Error("failed to get bb repo", "repo", repo, "err", err.Error())
			}
			cb(repo)
		}
	}

	if len(config.Projects) > 0 {
		for _, project := range config.Projects {
			if err := stashClient.WalkProjectRepositories(project, cb); err != nil {
				simplelog.Error("failed to walk bb project repos", "project", project, "err", err.Error())
			}
		}
	}

	if len(config.Projects) == 0 && len(config.Repos) == 0 {
		if err := stashClient.WalkRepositories(cb); err != nil {
			simplelog.Error("failed to walk bb repos", "err", err.Error())
		}
	}

	close(jobs)
	wg.Wait()

	return
}

func bbRepoWorker(stashClient stash.Stash, jobs <-chan stash.Repository, oldState *state.RepoState, newState *state.RepoState, wg *sync.WaitGroup) {
	defer wg.Done()
	for stashrepo := range jobs {
		if config.PrivateOnly && stashrepo.Public {
			simplelog.Info("Skip non private",
				"repo", stashrepo.FullName(), "parent", stashrepo.Origin.FullName())
			continue
		}

		if stashrepo.IsFork() && config.NoForks {
			simplelog.Info("Skip fork",
				"repo", stashrepo.FullName(), "parent", stashrepo.Origin.FullName())
			continue
		}

		repo := bbrepo.NewBitbucketRepo(stashrepo)
		repoState, ok := oldState.Load(repo.Name())
		if ok && repoState.Reference != "" {
			// Checks only for old or incomplete repo
			if !bbHaveNewCommits(stashClient, stashrepo, repoState.Reference) { // repoState.Reference
				simplelog.Info("Skip by commits filter", "repo", repo.Name())
				if newState != nil {
					newState.Store(repo.Name(), repoState.Reference)
				}
				continue
			}
		}

		if config.DryRun {
			fmt.Println(repo.ProjectURL())
		} else {
			if checker.CheckRepo(repo) && newState != nil {
				newState.Store(repo.Name(), repo.Reference())
			}
		}
	}
}

func bbHaveNewCommits(stashClient stash.Stash, repo stash.Repository, reference string) bool {
	haveChanges, err := stashClient.HaveCommitsSince(repo.Project.Key, repo.Name, reference)
	if err != nil {
		simplelog.Warn("failed to check since filter", "repo", repo.FullName(), "err", err.Error())
	}

	return haveChanges
}
