package commands

import (
	"fmt"
	"sync"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/hector/internal/checker"
	"a.yandex-team.ru/security/hector/internal/cli"
	"a.yandex-team.ru/security/hector/internal/config"
	"a.yandex-team.ru/security/hector/internal/dist"
	"a.yandex-team.ru/security/hector/internal/remote/distrepo"
	"a.yandex-team.ru/security/hector/internal/state"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

var distCmd = &cobra.Command{
	Use:   "dist [flags] -- /check-binary [child-flags]",
	Short: "Walk though dist repos",
	RunE:  cli.WrapCobraCommand("dist", runDistCmd),
}

func init() {
	RootCmd.AddCommand(distCmd)
}

func runDistCmd(oldState *state.RepoState, newState *state.RepoState) (err error) {
	if config.DryRun {
		fmt.Println("Acceptable packages: ")
	}

	// convert to bytes
	config.MaxRepoSize *= 1024

	distClient := dist.NewClient(config.DistBaseURL)
	jobs := make(chan dist.Package, config.Concurrency)
	var wg sync.WaitGroup
	wg.Add(config.Concurrency + 1)
	for w := 0; w < config.Concurrency+1; w++ {
		go distPackageWorker(jobs, oldState, newState, &wg)
	}

	total := 0
	cb := func(pkg dist.Package) bool {
		if config.MaxRepos > 0 && total >= config.MaxRepos {
			// stop iterate
			return false
		}

		total += 1
		jobs <- pkg
		return true
	}

	if len(config.Projects) > 0 {
		for _, project := range config.Projects {
			if err := distClient.WalkProject(project, cb); err != nil && err != dist.ErrStopWalk {
				simplelog.Error("failed to walk dist project", "project", project, "err", err.Error())
			}
		}
	}

	if len(config.Projects) == 0 && len(config.Repos) == 0 {
		if err := distClient.WalkProjects(cb); err != nil && err != dist.ErrStopWalk {
			simplelog.Error("failed to walk dist projects", "err", err.Error())
		}
	}

	close(jobs)
	wg.Wait()

	return
}

func distPackageWorker(jobs <-chan dist.Package, oldState *state.RepoState, newState *state.RepoState, wg *sync.WaitGroup) {
	defer wg.Done()
	for pkg := range jobs {
		if config.MaxRepoSize > 0 && pkg.Size > config.MaxRepoSize {
			simplelog.Info("Skip pkg due to size limit",
				"pkg_url", pkg.URL, "size", pkg.Size, "max-size", config.MaxRepoSize)
			continue
		}

		stateKey := pkg.FullName()
		repoState, ok := oldState.Load(stateKey)
		if ok && repoState.Reference == pkg.Version {
			simplelog.Info("Skip old pkg", "pkg_url", pkg.URL)
			if newState != nil {
				newState.Store(stateKey, repoState.Reference)
			}
			continue
		}

		if config.DryRun {
			fmt.Println(pkg.URL)
		} else {
			repo := distrepo.NewDistRepo(pkg)
			if checker.CheckRepo(repo) && newState != nil {
				newState.Store(repo.Name(), repo.Reference())
			}
		}
	}
}
