package commands

import (
	"context"
	"fmt"
	"net/url"
	"strings"
	"sync"
	"time"

	"github.com/cenkalti/backoff/v4"
	"github.com/google/go-github/v35/github"
	"github.com/spf13/cobra"
	"golang.org/x/oauth2"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/hector/internal/checker"
	"a.yandex-team.ru/security/hector/internal/cli"
	"a.yandex-team.ru/security/hector/internal/config"
	"a.yandex-team.ru/security/hector/internal/gitutils"
	"a.yandex-team.ru/security/hector/internal/remote/githubrepo"
	"a.yandex-team.ru/security/hector/internal/state"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

var ghCmd = &cobra.Command{
	Use:   "gh [flags] -- /check-binary [child-flags]",
	Short: "Walk though github.y-t.ru repos",
	RunE:  cli.WrapCobraCommand("gh", runGhCmd),
}

var ghRepoSet = sync.Map{}

func init() {
	RootCmd.AddCommand(ghCmd)
}

type ghListCb func(*github.Repository) (goAhead bool)

func runGhCmd(oldState *state.RepoState, newState *state.RepoState) (err error) {
	gitErr := gitutils.CheckVersion()
	if gitErr != nil {
		simplelog.Warn(gitErr.Error())
	}

	if config.DryRun {
		fmt.Println("Acceptable repos: ")
	}

	ctx := context.Background()
	ts := oauth2.StaticTokenSource(
		&oauth2.Token{AccessToken: config.GhToken},
	)
	tc := oauth2.NewClient(ctx, ts)
	ghClient := github.NewClient(tc)
	ghClient.BaseURL, _ = url.Parse(config.GHBaseURL)

	jobs := make(chan *github.Repository, config.Concurrency)
	var wg sync.WaitGroup
	wg.Add(config.Concurrency + 1)
	for w := 0; w < config.Concurrency+1; w++ {
		go ghRepoWorker(ghClient, jobs, oldState, newState, &wg)
	}

	total := 0
	cb := func(repo *github.Repository) bool {
		if config.MaxRepos > 0 && total >= config.MaxRepos {
			// stop iterate
			return false
		}
		total += 1
		jobs <- repo
		return true
	}

	if len(config.Repos) > 0 {
		for _, repo := range config.Repos {
			ghListRepo(ghClient, repo, cb)
		}
	}

	if len(config.Projects) > 0 {
		for _, project := range config.Projects {
			ghListProject(ghClient, project, cb)
		}
	}

	if len(config.Users) > 0 {
		for _, user := range config.Users {
			ghListUserRepo(ghClient, user, cb)
		}
	}

	if len(config.SearchQuery) != 0 {
		ghSearchRepos(ghClient, config.SearchQuery, cb)
	}

	if len(config.Repos)+len(config.Projects)+len(config.Users)+len(config.SearchQuery) == 0 {
		if config.PrivateOnly {
			ghListPrivate(ghClient, cb)
		} else {
			ghListPrivate(ghClient, cb)
			ghListAll(ghClient, cb)
		}
	}

	close(jobs)
	wg.Wait()

	return
}

func ghListAll(ghClient *github.Client, cb ghListCb) {
	ctx := context.Background()
	opts := &github.RepositoryListAllOptions{}
	for {
		repos, resp, err := ghClient.Repositories.ListAll(ctx, opts)
		if err != nil {
			simplelog.Error("failed to list repos", "error", err.Error())
			return
		}
		_ = resp.Body.Close()

		if len(repos) == 0 {
			return
		}

		for _, repo := range repos {
			// ListAll repos doesn't return clone urls, so we must to get info about repo one more time :(
			repo, resp, err := ghClient.Repositories.Get(ctx, repo.GetOwner().GetLogin(), repo.GetName())
			if err != nil {
				simplelog.Error("failed to get repo info, skip it", "repo", repo.GetFullName(), "error", err.Error())
				continue
			}
			_ = resp.Body.Close()

			if !cb(repo) {
				return
			}
		}
		opts.Since = *repos[len(repos)-1].ID
	}
}

func ghListPrivate(ghClient *github.Client, cb ghListCb) {
	ctx := context.Background()
	opts := &github.RepositoryListOptions{
		Type: "private",
	}

	for {
		repos, resp, err := ghClient.Repositories.List(ctx, "", opts)
		if err != nil {
			simplelog.Error("failed to list user repos", "user", "self", "error", err.Error())
			return
		}
		_ = resp.Body.Close()

		if len(repos) == 0 {
			return
		}

		for _, repo := range repos {
			if !cb(repo) {
				return
			}
		}

		if resp.NextPage == 0 {
			return
		}

		opts.ListOptions.Page = resp.NextPage
	}
}

func ghListProject(ghClient *github.Client, project string, cb ghListCb) {
	ctx := context.Background()
	opts := &github.RepositoryListByOrgOptions{}
	if config.PrivateOnly {
		opts.Type = "private"
	}

	for {
		repos, resp, err := ghClient.Repositories.ListByOrg(ctx, project, opts)
		if err != nil {
			simplelog.Error("failed to list project repos", "project", project, "error", err.Error())
			return
		}
		_ = resp.Body.Close()

		if len(repos) == 0 {
			return
		}

		for _, repo := range repos {
			if !cb(repo) {
				return
			}
		}

		if resp.NextPage == 0 {
			return
		}

		opts.ListOptions.Page = resp.NextPage
	}
}

func ghListUserRepo(ghClient *github.Client, user string, cb ghListCb) {
	ctx := context.Background()
	opts := &github.RepositoryListOptions{}

	for {
		repos, resp, err := ghClient.Repositories.List(ctx, user, opts)
		if err != nil {
			simplelog.Error("failed to list user repos", "user", user, "error", err.Error())
			return
		}
		_ = resp.Body.Close()

		if len(repos) == 0 {
			return
		}

		for _, repo := range repos {
			if !cb(repo) {
				return
			}
		}

		if resp.NextPage == 0 {
			return
		}

		opts.ListOptions.Page = resp.NextPage
	}
}

func ghListRepo(ghClient *github.Client, repoPath string, cb ghListCb) {
	ctx := context.Background()
	repoInfo := strings.SplitN(repoPath, "/", 2)
	repo, resp, err := ghClient.Repositories.Get(ctx, repoInfo[0], repoInfo[1])
	if err != nil {
		simplelog.Error("failed to get repo", "repo", repoPath, "error", err.Error())
		return
	}

	_ = resp.Body.Close()
	cb(repo)
}

func ghRepoWorker(ghClient *github.Client, jobs <-chan *github.Repository, oldState *state.RepoState, newState *state.RepoState, wg *sync.WaitGroup) {
	defer wg.Done()
	for ghRepo := range jobs {
		if config.PrivateOnly && !ghRepo.GetPrivate() {
			simplelog.Info("Skip non private",
				"repo", ghRepo.GetFullName(), "parent", ghRepo.GetParent().GetFullName())
			continue
		}

		if config.NoForks && ghRepo.GetFork() {
			simplelog.Info("Skip fork",
				"repo", ghRepo.GetFullName(), "parent", ghRepo.GetParent().GetFullName())
			continue
		}

		if config.MaxRepoSize > 0 && ghRepo.GetSize() > config.MaxRepoSize {
			simplelog.Info("Skip repo due to size limit",
				"repo", ghRepo.GetFullName(), "size", ghRepo.GetSize(), "max-size", config.MaxRepoSize)
			continue
		}

		if !config.UseSSH && ghRepo.GetPrivate() {
			simplelog.Info("Skip private, allow SSH to work with private repos", "repo", ghRepo.GetFullName())
			continue
		}

		_, exists := ghRepoSet.LoadOrStore(ghRepo.GetFullName(), nil)
		if exists {
			simplelog.Debug("Skip non-uniq repo", "repo", ghRepo.GetFullName())
			continue
		}

		repoState, ok := oldState.Load(ghRepo.GetFullName())
		if ok && repoState.Reference != "" {
			// Checks only for old or incomplete repo
			if !ghHaveNewCommits(ghClient, ghRepo, repoState.Reference) { // repoState.Reference
				simplelog.Info("Skip by commits filter", "repo", ghRepo.GetFullName())
				if newState != nil {
					newState.Store(ghRepo.GetFullName(), repoState.Reference)
				}
				continue
			}
		}

		if !ghIsLanguageAcceptable(ghClient, ghRepo) {
			simplelog.Info("Skip by language filter", "repo", ghRepo.GetFullName())
			continue
		}

		repo := githubrepo.NewGithubRepo(ghRepo)
		if config.DryRun {
			fmt.Println(repo.ProjectURL())
		} else {
			if checker.CheckRepo(repo) && newState != nil {
				newState.Store(ghRepo.GetFullName(), repo.Reference())
			}
		}
	}
}

func ghIsLanguageAcceptable(ghClient *github.Client, repo *github.Repository) bool {
	if len(config.Languages) == 0 {
		return true
	}

	languages, _, err := ghClient.Repositories.ListLanguages(context.Background(), repo.GetOwner().GetLogin(), repo.GetName())
	if len(languages) == 0 || err != nil {
		return true
	}

	total := 0
	for _, lines := range languages {
		total += lines
	}

	for l, lines := range languages {
		lang := strings.ToLower(l)
		if min, ok := config.Languages[lang]; ok {
			if min == 0 {
				return true
			}

			if lines > 0 && total/lines*100 >= min {
				return true
			}

		}
	}
	return false
}

func ghHaveNewCommits(ghClient *github.Client, repo *github.Repository, reference string) bool {
	opts := github.CommitsListOptions{
		ListOptions: github.ListOptions{
			Page:    1,
			PerPage: 1,
		},
	}

	commits, _, err := ghClient.Repositories.ListCommits(context.Background(), repo.GetOwner().GetLogin(), repo.GetName(), &opts)
	if err != nil {
		simplelog.Warn("failed to check new commits", "repo", repo.GetFullName(), "err", err.Error())
		return true
	}

	return len(commits) > 0 && commits[0].GetSHA() != reference
}

func ghSearchRepos(ghClient *github.Client, query string, cb ghListCb) {
	ctx := context.Background()
	opts := &github.SearchOptions{
		Sort:  "indexed",
		Order: "desc",
	}

	zeroBackoff := new(backoff.ZeroBackOff)
	for {
		var result *github.CodeSearchResult
		var resp *github.Response

		// TODO(buglloc): WTF?! Just write a correct retrier, asshole
		err := backoff.Retry(
			func() (err error) {
				result, resp, err = ghClient.Search.Code(ctx, query, opts)
				if err != nil {
					var abuseErr *github.AbuseRateLimitError
					if xerrors.As(err, &abuseErr) {
						simplelog.Warn(abuseErr.Error(), "delay", abuseErr.GetRetryAfter().String())
						time.Sleep(abuseErr.GetRetryAfter())
					}
					return err
				}

				_ = resp.Body.Close()
				return nil
			},
			backoff.WithMaxRetries(zeroBackoff, 5),
		)

		if err != nil {
			simplelog.Error("failed to search gh repos", "error", err.Error())
			return
		}

		if len(result.CodeResults) == 0 {
			return
		}

		for _, codeRes := range result.CodeResults {
			if codeRes.Repository == nil || codeRes.Repository.Owner == nil {
				continue
			}

			repo := codeRes.Repository
			_, exists := ghRepoSet.LoadOrStore(repo.GetFullName(), nil)
			if exists {
				simplelog.Debug("Skip non-uniq repo", "repo", repo.GetFullName())
				continue
			}

			// Search repos doesn't return clone urls, so we must to get info about repo one more time :(
			ghRepo, resp, err := ghClient.Repositories.Get(ctx, repo.GetOwner().GetLogin(), repo.GetName())
			if err != nil {
				simplelog.Error("failed to get repo", "repo", repo.GetFullName(), "error", err.Error())
				continue
			}

			_ = resp.Body.Close()
			if !cb(ghRepo) {
				return
			}
		}

		if resp.NextPage == 0 {
			return
		}

		opts.ListOptions.Page = resp.NextPage
	}
}
