package commands

import (
	"context"
	"errors"
	"net/url"
	"strings"
	"sync"

	"github.com/google/go-github/v35/github"
	"github.com/spf13/cobra"
	"golang.org/x/oauth2"

	"a.yandex-team.ru/security/hector/internal/checker"
	"a.yandex-team.ru/security/hector/internal/cli"
	"a.yandex-team.ru/security/hector/internal/config"
	"a.yandex-team.ru/security/hector/internal/gitutils"
	"a.yandex-team.ru/security/hector/internal/remote/gistrepo"
	"a.yandex-team.ru/security/hector/internal/state"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

var gistCmd = &cobra.Command{
	Use:   "gist [flags] -- /check-binary [child-flags]",
	Short: "Walk though gists",
	RunE:  cli.WrapCobraCommand("gist", runGistCmd),
}

func init() {
	RootCmd.AddCommand(gistCmd)
}

type gistListCb func(gist *github.Gist) bool

func runGistCmd(oldState *state.RepoState, newState *state.RepoState) (err error) {
	gitErr := gitutils.CheckVersion()
	if gitErr != nil {
		simplelog.Warn(gitErr.Error())
	}

	if config.DryRun {
		return errors.New("gist doesn't hve dry-run mode")
	}

	ctx := context.Background()
	ts := oauth2.StaticTokenSource(
		&oauth2.Token{AccessToken: config.GhToken},
	)
	tc := oauth2.NewClient(ctx, ts)
	ghClient := github.NewClient(tc)
	ghClient.BaseURL, _ = url.Parse(config.GHBaseURL)

	jobs := make(chan *github.Gist, config.Concurrency)
	var wg sync.WaitGroup
	for w := 0; w < config.Concurrency+1; w++ {
		go ghGistWorker(jobs, oldState, newState, &wg)
	}

	total := 0
	cb := func(gist *github.Gist) bool {
		if config.MaxRepos > 0 && total >= config.MaxRepos {
			// stop iterate
			return false
		}
		total += 1
		jobs <- gist
		wg.Add(1)

		return true
	}

	if len(config.Users) > 0 {
		for _, user := range config.Users {
			if err := gistListUser(ghClient, user, cb); err != nil {
				simplelog.Error("failed to list user gists", "user", user, "error", err.Error())
			}
		}
	} else {
		if err := gistListAll(ghClient, cb); err != nil {
			simplelog.Error("failed to list gists", "error", err.Error())
		}
	}

	close(jobs)
	wg.Wait()

	return
}

func gistListAll(ghClient *github.Client, cb gistListCb) error {
	ctx := context.Background()
	opts := &github.GistListOptions{}
	for {
		gists, resp, err := ghClient.Gists.ListAll(ctx, opts)
		if err != nil {
			return err
		}
		_ = resp.Body.Close()

		if len(gists) == 0 {
			return nil
		}

		for _, gist := range gists {
			if !cb(gist) {
				return nil
			}
		}
		if resp.NextPage == 0 {
			break
		}

		opts.Page = resp.NextPage
	}
	return nil
}

func gistListUser(ghClient *github.Client, user string, cb gistListCb) error {
	ctx := context.Background()
	opts := &github.GistListOptions{}
	for {
		gists, resp, err := ghClient.Gists.List(ctx, user, opts)
		if err != nil {
			return err
		}
		_ = resp.Body.Close()

		if len(gists) == 0 {
			return nil
		}

		for _, gist := range gists {
			if !cb(gist) {
				return nil
			}
		}
		if resp.NextPage == 0 {
			break
		}

		opts.Page = resp.NextPage
	}
	return nil
}

func ghGistWorker(jobs <-chan *github.Gist, oldState *state.RepoState, newState *state.RepoState, wg *sync.WaitGroup) {
	for gist := range jobs {
		func() {
			defer wg.Done()

			if len(gist.Files) == 0 {
				simplelog.Info("Skip empty gist", "gist", gist.GetID())
				return
			}

			if !ghGistLanguageAcceptable(gist) {
				simplelog.Info("Skip by language filter", "gist", gist.GetID())
				return
			}

			gistState, ok := oldState.Load(gist.GetID())
			if ok && gistState.Reference != "" && gistState.Reference == gist.GetUpdatedAt().String() {
				simplelog.Info("Skip by commits filter", "gist", gist.GetID())
				return
			}

			repo := gistrepo.NewGistRepo(gist)
			if checker.CheckRepo(repo) && newState != nil {
				newState.Store(gist.GetID(), gist.GetUpdatedAt().String())
			}
		}()
	}
}

func ghGistLanguageAcceptable(gist *github.Gist) bool {
	if len(config.Languages) == 0 {
		return true
	}

	for _, gistFile := range gist.Files {
		lang := strings.ToLower(gistFile.GetLanguage())
		if _, ok := config.Languages[lang]; ok {
			return true
		}
	}
	return false
}
