package commands

import (
	"errors"
	"fmt"
	"net/url"
	"sync"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/hector/internal/checker"
	"a.yandex-team.ru/security/hector/internal/cli"
	"a.yandex-team.ru/security/hector/internal/config"
	"a.yandex-team.ru/security/hector/internal/gitlab"
	"a.yandex-team.ru/security/hector/internal/gitutils"
	"a.yandex-team.ru/security/hector/internal/remote/gitlabrepo"
	"a.yandex-team.ru/security/hector/internal/state"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

var gitlabCmd = &cobra.Command{
	Use:   "gitlab [flags] -- /check-binary [child-flags]",
	Short: "Walk though gitlab repos",
	RunE:  cli.WrapCobraCommand("gitlab", runGitlabCmd),
}

func init() {
	RootCmd.AddCommand(gitlabCmd)
}

func runGitlabCmd(oldState *state.RepoState, newState *state.RepoState) (err error) {
	gitErr := gitutils.CheckVersion()
	if gitErr != nil {
		simplelog.Warn(gitErr.Error())
	}

	if config.DryRun {
		fmt.Println("Acceptable repos: ")
	}

	gitlabURL, _ := url.Parse(config.GitlabBaseURL)
	gitlabClient := gitlab.NewClient(config.GitlabToken, gitlabURL)

	jobs := make(chan gitlab.Project, config.Concurrency)
	var wg sync.WaitGroup
	wg.Add(config.Concurrency + 1)
	for w := 0; w < config.Concurrency+1; w++ {
		go gitlabWorker(gitlabClient, jobs, oldState, newState, &wg)
	}

	total := 0
	cb := func(repo gitlab.Project) bool {
		if config.MaxRepos > 0 && total >= config.MaxRepos {
			// stop iterate
			return false
		}

		total += 1
		jobs <- repo
		return true
	}

	if len(config.Repos) > 0 {
		for _, repo := range config.Repos {
			repo, err := gitlabClient.GetProject(repo)
			if err != nil {
				simplelog.Error("failed to get gitlab repo", "repo", repo, "err", err.Error())
			}
			cb(repo)
		}
	}

	if len(config.Projects) > 0 {
		return errors.New("projects is unsupported for gitlab, use repos")
	}

	if len(config.Projects) == 0 && len(config.Repos) == 0 {
		if err := gitlabClient.WalkProjects(cb); err != nil {
			simplelog.Error("failed to walk gitlab repos", "err", err.Error())
		}
	}

	close(jobs)
	wg.Wait()

	return
}

func gitlabWorker(gitlabClient *gitlab.Client, jobs <-chan gitlab.Project, oldState *state.RepoState, newState *state.RepoState, wg *sync.WaitGroup) {
	defer wg.Done()
	for project := range jobs {
		if config.PrivateOnly && project.Visibility == gitlab.VisibilityPrivate {
			simplelog.Info("Skip non private",
				"repo", project.FullName())
			continue
		}

		if project.IsFork() && config.NoForks {
			simplelog.Info("Skip fork",
				"repo", project.FullName())
			continue
		}

		if !config.UseSSH && project.Visibility != gitlab.VisibilityPublic {
			simplelog.Info("Skip private, allow SSH to work with non-public repos", "repo", project.FullName())
			continue
		}

		repo := gitlabrepo.NewRepo(project)
		repoState, ok := oldState.Load(repo.Name())
		if ok && repoState.Reference != "" {
			// Checks only for old or incomplete repo
			if !gitlabHaveNewCommits(gitlabClient, project, repoState.Reference) { // repoState.Reference
				simplelog.Info("Skip by commits filter", "repo", repo.Name())
				if newState != nil {
					newState.Store(repo.Name(), repoState.Reference)
				}
				continue
			}
		}

		if config.DryRun {
			fmt.Println(repo.ProjectURL())
		} else {
			if checker.CheckRepo(repo) && newState != nil {
				newState.Store(repo.Name(), repo.Reference())
			}
		}
	}
}

func gitlabHaveNewCommits(gitlabClient *gitlab.Client, project gitlab.Project, reference string) bool {
	haveChanges, err := gitlabClient.HaveCommitsSince(project.Name, reference)
	if err != nil {
		simplelog.Warn("failed to check since filter", "repo", project.FullName(), "err", err.Error())
	}

	return haveChanges
}
