package commands

import (
	"errors"
	"io/ioutil"
	"regexp"
	"strconv"
	"sync"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/hector/internal/checker"
	"a.yandex-team.ru/security/hector/internal/cli"
	"a.yandex-team.ru/security/hector/internal/config"
	"a.yandex-team.ru/security/hector/internal/core"
	"a.yandex-team.ru/security/hector/internal/remote/pasterepo"
	"a.yandex-team.ru/security/hector/internal/state"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/libs/go/yahttp"
)

var pasteCmd = &cobra.Command{
	Use:   "paste [flags] -- /check-binary [child-flags]",
	Short: "Walk though paste.y-t.ru",
	RunE:  cli.WrapCobraCommand("paste", runPasteCmd),
}

func init() {
	RootCmd.AddCommand(pasteCmd)
}

func runPasteCmd(oldState *state.RepoState, newState *state.RepoState) (err error) {
	if config.DryRun {
		return errors.New("paste doesn't hve dry-run mode")
	}

	total := 0
	latest, err := getLatestID()
	if err != nil {
		simplelog.Crit(err.Error())
		return
	}

	jobs := make(chan int, config.Concurrency)
	var wg sync.WaitGroup
	for w := 0; w < config.Concurrency+1; w++ {
		go pasteWorker(jobs, newState, &wg)
	}

	// First of all checks incomplete pastes
	if oldState != nil {
		for name := range oldState.Info {
			id, err := strconv.Atoi(name)
			if err != nil {
				simplelog.Error("faield to parse incomplete paste", "name", name, "err", err.Error())
				continue
			}

			if !processPaste(id, jobs, &total, &wg) {
				break
			}
		}
	}

	// Now checks new pastes
	current := 0
	if oldState != nil {
		current = int(oldState.Latest)
	}

	for ; current <= latest; current++ {
		if !processPaste(current, jobs, &total, &wg) {
			break
		}
	}

	close(jobs)
	wg.Wait()
	newState.Latest = int64(current)

	return
}

func processPaste(id int, jobs chan<- int, total *int, wg *sync.WaitGroup) bool {
	if config.MaxRepos > 0 && *total >= config.MaxRepos {
		return false
	}

	*total += 1
	jobs <- id
	wg.Add(1)
	return true
}

func pasteWorker(jobs <-chan int, st *state.RepoState, wg *sync.WaitGroup) {
	for id := range jobs {
		func() {
			defer wg.Done()
			repo := pasterepo.NewPasteRepo(id)
			if !checker.CheckRepo(repo) && st != nil {
				st.Store(repo.Name(), repo.Reference())
			}
		}()
	}
}

func getLatestID() (last int, err error) {
	resp, err := core.Client.Get("https://paste.yandex-team.ru/last")
	if err != nil {
		return
	}
	defer yahttp.GracefulClose(resp.Body)

	if resp.StatusCode != 200 {
		err = errors.New("failed to get last pastes")
		return
	}

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return
	}

	re := regexp.MustCompile(`<li>№ <a href="/\d+">(\d+)</a>.</li>`)
	submatches := re.FindSubmatch(body)
	if len(submatches) == 0 {
		err = xerrors.Errorf("failed to parse response: %q", string(body))
		return
	}

	last, err = strconv.Atoi(string(submatches[1]))
	return
}
