package commands

import (
	"fmt"
	"sync"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/hector/internal/checker"
	"a.yandex-team.ru/security/hector/internal/cli"
	"a.yandex-team.ru/security/hector/internal/config"
	"a.yandex-team.ru/security/hector/internal/remote/pypirepo"
	"a.yandex-team.ru/security/hector/internal/state"
	"a.yandex-team.ru/security/libs/go/pypi"
	"a.yandex-team.ru/security/libs/go/pypi/repos/pypilocalshop"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

var pypiCmd = &cobra.Command{
	Use:   "pypi [flags] -- /check-binary [child-flags]",
	Short: "Walk though pypi.y-t.ru local packages",
	RunE:  cli.WrapCobraCommand("pypi", runPypiCmd),
}

func init() {
	RootCmd.AddCommand(pypiCmd)
}

func runPypiCmd(oldState *state.RepoState, newState *state.RepoState) (err error) {
	pypiClient, err := pypilocalshop.New(pypilocalshop.Options{
		BaseURL:   "https://pypi.yandex-team.ru",
		LocalOnly: true,
		Login:     config.PypiLogin,
		Password:  config.PypiPassword,
		Name:      "pypi-local",
	})
	if err != nil {
		return err
	}

	jobs := make(chan pypi.Package, config.Concurrency)
	var wg sync.WaitGroup
	wg.Add(config.Concurrency + 1)
	for w := 0; w < config.Concurrency+1; w++ {
		go pypiPackageWorker(jobs, oldState, newState, &wg)
	}

	total := 0
	for pypiClient.Next() {
		pkg, err := pypiClient.Package()
		if err != nil {
			simplelog.Error("failed to get package", "err", err)
			continue
		}

		if config.MaxRepos > 0 && total >= config.MaxRepos {
			// stop iterate
			break
		}

		total += 1
		jobs <- pkg
	}

	err = pypiClient.Error()

	close(jobs)
	wg.Wait()

	return
}

func pypiPackageWorker(jobs <-chan pypi.Package, oldState *state.RepoState, newState *state.RepoState, wg *sync.WaitGroup) {
	defer wg.Done()
	for pkg := range jobs {
		if err := pkg.Resolve(); err != nil {
			simplelog.Error("failed to resolve package", "err", err)
			continue
		}

		for _, releaseDetails := range pkg.Releases() {
			if len(releaseDetails) == 0 {
				continue
			}

			release := releaseDetails[0]
			stateKey := pkg.Name() + release.Version
			if _, ok := oldState.Load(stateKey); ok {
				simplelog.Info("Skip old pkg", "pkg", pkg.Name())
				if newState != nil {
					newState.Store(stateKey, release.Version)
				}
				continue
			}

			if config.DryRun {
				fmt.Println(release.DownloadURL)
			} else {
				repo := pypirepo.NewPypiRepo(pkg, release)
				if checker.CheckRepo(repo) && newState != nil {
					newState.Store(stateKey, repo.Reference())
				}
			}
		}
	}
}
