package commands

import (
	"errors"
	"fmt"
	"os"
	"os/user"
	"path/filepath"
	"strconv"
	"strings"

	"github.com/google/go-github/v35/github"
	"github.com/spf13/cobra"
	"github.com/spf13/pflag"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/hector/internal/config"
	"a.yandex-team.ru/security/hector/internal/fsutils"
)

type Result struct {
	InternalName string
	Repo         *github.Repository
}

var RootCmd = &cobra.Command{
	Use:          "hector",
	SilenceUsage: true,
	Short:        "Repo walker",
	PersistentPreRunE: func(cmd *cobra.Command, args []string) error {
		if cmd == versionCmd {
			// Skip pre run for version command
			return nil
		}

		flags := cmd.Flags()
		if showVer, err := flags.GetBool("version"); err == nil && showVer {
			fmt.Println(config.FullVersion())
			os.Exit(0)
		}

		cfgPath, _ := flags.GetString("config")
		cfg, err := loadConfig(cfgPath)
		if err != nil {
			return err
		}

		if len(args) > 0 {
			config.ChildBinary = args[0]
			config.ChildArgs = args[1:]
		} else if len(cfg.CheckCommand) > 0 {
			config.ChildBinary = cfg.CheckCommand[0]
			config.ChildArgs = cfg.CheckCommand[1:]
		} else {
			return errors.New("please provide binary for repo checks")
		}

		//TODO(buglloc): UGLY!!!11
		config.Languages = parseLanguagesFilter(flags, cfg)
		if flags.Lookup("bb-url").Changed {
			config.BBBaseURL, _ = flags.GetString("bb-url")
		} else {
			config.BBBaseURL = cfg.BbURL
		}

		if flags.Lookup("gh-url").Changed {
			config.GHBaseURL, _ = flags.GetString("gh-url")
		} else {
			config.GHBaseURL = cfg.GhURL
		}

		if flags.Lookup("gitlab-url").Changed {
			config.GitlabBaseURL, _ = flags.GetString("gitlab-url")
		}

		if flags.Lookup("gh-token").Changed {
			config.GhToken, _ = flags.GetString("gh-token")
		} else {
			config.GhToken = cfg.GhToken
		}

		if flags.Lookup("bb-login").Changed {
			config.BBLogin, _ = flags.GetString("bb-login")
		} else {
			config.BBLogin = cfg.BBLogin
		}

		if flags.Lookup("bb-password").Changed {
			config.BBPassword, _ = flags.GetString("bb-password")
		} else {
			config.BBPassword = cfg.BBPassword
		}

		if flags.Lookup("gitlab-token").Changed {
			config.GitlabToken, _ = flags.GetString("gitlab-token")
		} else {
			config.GitlabToken = cfg.GitlabToken
		}

		if flags.Lookup("pypi-login").Changed {
			config.PypiLogin, _ = flags.GetString("pypi-login")
		} else {
			config.PypiLogin = cfg.PypiLogin
		}

		if flags.Lookup("pypi-password").Changed {
			config.PypiPassword, _ = flags.GetString("pypi-password")
		} else {
			config.PypiPassword = cfg.PypiPassword
		}

		if flags.Lookup("result").Changed {
			config.ResultDir, _ = flags.GetString("result")
		} else {
			config.ResultDir = cfg.ResultDir
		}

		if flags.Lookup("work").Changed {
			config.WorkDir, _ = flags.GetString("work")
		} else {
			config.WorkDir = cfg.WorkDir
		}

		if flags.Lookup("timeout").Changed {
			config.Timeout, _ = flags.GetInt("timeout")
		} else {
			config.Timeout = cfg.Timeout
		}

		if flags.Lookup("concurrency").Changed {
			config.Concurrency, _ = flags.GetInt("concurrency")
		} else {
			config.Concurrency = cfg.Concurrency
		}

		if flags.Lookup("count-limit").Changed {
			config.MaxRepos, _ = flags.GetInt("count-limit")
		} else {
			config.MaxRepos = cfg.CountLimit
		}

		if flags.Lookup("size-limit").Changed {
			config.MaxRepoSize, _ = flags.GetInt("size-limit")
		} else {
			config.MaxRepoSize = cfg.SizeLimit
		}

		if flags.Lookup("parse-stdout").Changed {
			config.ParseStdout, _ = flags.GetBool("parse-stdout")
		} else {
			config.ParseStdout = cfg.ParseStdout
		}

		if flags.Lookup("use-ssh").Changed {
			config.UseSSH, _ = flags.GetBool("use-ssh")
		} else {
			config.UseSSH = cfg.UseSSH
		}

		if flags.Lookup("state-in").Changed {
			config.StateIn, _ = flags.GetString("state-in")
		} else {
			config.StateIn = cfg.StateIn
		}

		if flags.Lookup("state-out").Changed {
			config.StateOut, _ = flags.GetString("state-out")
		} else {
			config.StateOut = cfg.StateOut
		}

		if flags.Lookup("no-forks").Changed {
			config.NoForks, _ = flags.GetBool("no-forks")
		} else {
			config.NoForks = cfg.NoForks
		}

		if flags.Lookup("dry-run").Changed {
			config.DryRun, _ = flags.GetBool("dry-run")
		} else {
			config.DryRun = cfg.DryRun
		}

		if flags.Lookup("private-only").Changed {
			config.PrivateOnly, _ = flags.GetBool("private-only")
		} else {
			config.PrivateOnly = cfg.PrivateOnly
		}

		if flags.Lookup("repos").Changed {
			config.Repos, _ = flags.GetStringSlice("repos")
		} else {
			config.Repos = cfg.Repos
		}

		if flags.Lookup("projects").Changed {
			config.Projects, _ = flags.GetStringSlice("projects")
		} else {
			config.Projects = cfg.Projects
		}

		if flags.Lookup("users").Changed {
			config.Users, _ = flags.GetStringSlice("users")
		} else {
			config.Users = cfg.Users
		}

		if flags.Lookup("search-query").Changed {
			config.SearchQuery, _ = flags.GetString("search-query")
		} else {
			config.SearchQuery = cfg.SearchQuery
		}

		_ = fsutils.DirMustExists(config.ResultDir)
		_ = fsutils.DirMustExists(config.WorkDir)
		return nil
	},
}

func Execute() {
	RootCmd.Version = config.FullVersion()
	RootCmd.SetVersionTemplate("{{.Version}}\n")

	if err := RootCmd.Execute(); err != nil {
		os.Exit(1)
	}
}

func init() {
	flags := RootCmd.PersistentFlags()
	flags.String("config", "",
		"config file (default is $HOME/.hector.toml)")
	flags.String("work", config.WorkDir,
		"work directory")
	flags.String("result", config.ResultDir,
		"results directory")
	flags.StringSlice("projects", config.Projects,
		"list of projects (organizations) to analyze (e.g. security)")
	flags.StringSlice("repos", config.Repos,
		"list of repos to analyze (e.g. security/hector)")
	flags.StringSlice("users", config.Users,
		"list of users to analyze (e.g. security/hector)")
	flags.Int("count-limit", config.MaxRepos,
		"limit repos count (0 to unlimited)")
	flags.Int("size-limit", config.MaxRepoSize,
		"limit repo size in Kb (0 to unlimited)")
	flags.Int("timeout", config.Timeout,
		"timeout in sec (0 to unlimited)")
	flags.Int("concurrency", config.Concurrency,
		"number of threads")
	flags.Bool("parse-stdout", config.ParseStdout,
		"parse&enrich tool data (must be in hector format)")
	flags.Bool("no-forks", config.NoForks,
		"skip forks")
	flags.Bool("dry-run", config.DryRun,
		"dry run, just print all acceptable repos")
	flags.Bool("private-only", config.PrivateOnly,
		"check only private repos")
	flags.Bool("use-ssh", config.UseSSH,
		"use SSH to clone repos, allow to work with private repos")
	flags.String("gh-token", "",
		"github OAuth token")
	flags.String("bb-login", "",
		"bitbucket login")
	flags.String("bb-password", "",
		"bitbucket password")
	flags.String("pypi-login", "",
		"pypi login")
	flags.String("pypi-password", "",
		"pypi password")
	flags.StringSlice("languages", nil,
		"language filter")
	flags.String("state-in", config.StateIn,
		"load checks state from file")
	flags.String("state-out", config.StateOut,
		"save checks state to file")
	flags.String("dist-url", config.DistBaseURL,
		"base dist url")
	flags.String("bb-url", config.BBBaseURL,
		"base url for BitBucket API")
	flags.String("gh-url", config.GHBaseURL,
		"base url for Github API")
	flags.String("gitlab-url", config.GitlabBaseURL,
		"Gitlab host")
	flags.String("gitlab-token", config.GitlabToken,
		"Gitlab auth token")
	flags.String("search-query", "",
		"search query to search for thought repo API")
}

func loadConfig(cfgFile string) (*config.Config, error) {
	if cfgFile != "" {
		_, err := os.Stat(cfgFile)
		if err != nil {
			return nil, xerrors.Errorf("failed to open config file: %s", err)
		}
	} else {
		u, err := user.Current()
		if err != nil {
			return nil, xerrors.Errorf("failed to get current user: %w", err)
		}

		if u.HomeDir == "" {
			return nil, xerrors.Errorf("empty user %q homedir", u.Username)
		}

		cfgFile = filepath.Join(u.HomeDir, ".hector.toml")
	}

	return config.LoadConfig(cfgFile)
}

func parseLanguagesFilter(flags *pflag.FlagSet, cfg *config.Config) map[string]int {
	var filter []string
	if flags.Lookup("languages").Changed {
		filter, _ = flags.GetStringSlice("languages")
	} else {
		filter = cfg.Languages
	}

	result := make(map[string]int)
	for _, f := range filter {
		sp := strings.SplitN(f, ":", 2)
		lang := sp[0]
		min := 0
		if len(sp) == 2 {
			if val, err := strconv.Atoi(sp[1]); err == nil {
				min = val
			}
		}
		result[lang] = min
	}
	return result
}
