package config

import (
	"context"
	"os"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend"
	"github.com/heetch/confita/backend/env"
	"github.com/heetch/confita/backend/file"

	"a.yandex-team.ru/library/go/core/buildinfo"
)

const (
	ProductName = "hector"
	Version     = "0.11"
	RetryCount  = 3
)

var (
	DistBaseURL   = "http://dist.yandex.net"
	BBBaseURL     = "https://bb.yandex-team.ru"
	GHBaseURL     = "https://github.yandex-team.ru/api/v3/"
	GitlabBaseURL = ""
	GhToken       string
	BBLogin       string
	BBPassword    string
	GitlabToken   string
	PypiLogin     string
	PypiPassword  string
	MaxRepos      = 0
	Repos         []string
	Projects      []string
	Users         []string
	SearchQuery   string
	Concurrency   = 16
	MaxRepoSize   = 1024 * 1024 // KiB
	WorkDir       = "/tmp/code"
	ResultDir     = "/tmp/results"
	ParseStdout   = false
	UseSSH        = false
	Timeout       = 300
	ChildBinary   string
	ChildArgs     []string
	Languages     map[string]int
	StateIn       string
	StateOut      string
	NoForks       = false
	DryRun        = false
	PrivateOnly   = false

	fullVersion string
)

func FullVersion() string {
	if fullVersion != "" {
		return fullVersion
	}

	if buildinfo.Info.ArcadiaSourceRevision != "" {
		fullVersion = Version + "." + buildinfo.Info.ArcadiaSourceRevision
	} else {
		fullVersion = Version
	}

	return fullVersion
}

type (
	Config struct {
		WorkDir      string   `config:"work" toml:"Work"`
		ResultDir    string   `config:"result" toml:"Result"`
		Projects     []string `config:"projects" toml:"Projects"`
		Repos        []string `config:"repos" toml:"Repos"`
		Users        []string `config:"users" toml:"Users"`
		SearchQuery  string   `config:"search-query" toml:"SearchQuery"`
		CountLimit   int      `config:"count-limit" toml:"CountLimit"`
		SizeLimit    int      `config:"size-limit" toml:"SizeLimit"`
		Timeout      int      `config:"timeout" toml:"Timeout"`
		Concurrency  int      `config:"concurrency" toml:"Concurrency"`
		ParseStdout  bool     `config:"parse-stdout" toml:"ParseStdout"`
		NoForks      bool     `config:"no-forks" toml:"NoForks"`
		DryRun       bool     `config:"dry-run" toml:"DryRun"`
		PrivateOnly  bool     `config:"private-only" toml:"PrivateOnly"`
		UseSSH       bool     `config:"use-ssh" toml:"UseSsh"`
		CheckCommand []string `config:"check-command" toml:"CheckCommand"`
		GhToken      string   `config:"gh-token" toml:"GhToken"`
		BBLogin      string   `config:"bb-login" toml:"BbLogin"`
		BBPassword   string   `config:"bb-password" toml:"BbPassword"`
		GitlabToken  string   `config:"gitlab-token" toml:"GitlabToken"`
		PypiLogin    string   `config:"pypi-login" toml:"PypiLogin"`
		PypiPassword string   `config:"pypi-password" toml:"PypiPassword"`
		Languages    []string `config:"languages" toml:"Languages"`
		StateIn      string   `config:"state-in" toml:"StateIn"`
		StateOut     string   `config:"state-out" toml:"StateOut"`
		DistURL      string   `config:"dist-url" toml:"DistUrl"`
		BbURL        string   `config:"bb-url" toml:"BbUrl"`
		GhURL        string   `config:"gh-url" toml:"GhUrl"`
		GitlabURL    string   `config:"gitlab-url" toml:"GitlabUrl"`
	}
)

func LoadConfig(cfgPath string) (*Config, error) {
	var backends []backend.Backend
	if _, err := os.Stat(cfgPath); err == nil {
		// that's fine
		backends = append(backends, file.NewBackend(cfgPath))
	}
	backends = append(backends, env.NewBackend())

	loader := confita.NewLoader(backends...)

	cfg := &Config{
		WorkDir:      WorkDir,
		ResultDir:    ResultDir,
		Projects:     Projects,
		Repos:        Repos,
		Users:        Users,
		CountLimit:   MaxRepos,
		SizeLimit:    MaxRepoSize,
		Timeout:      Timeout,
		Concurrency:  Concurrency,
		ParseStdout:  ParseStdout,
		NoForks:      NoForks,
		DryRun:       DryRun,
		PrivateOnly:  PrivateOnly,
		UseSSH:       UseSSH,
		GhToken:      GhToken,
		BBLogin:      BBBaseURL,
		BBPassword:   BBPassword,
		GitlabToken:  GitlabToken,
		PypiLogin:    PypiLogin,
		PypiPassword: PypiPassword,
		StateIn:      StateIn,
		StateOut:     StateOut,
		DistURL:      DistBaseURL,
		BbURL:        BBBaseURL,
		GhURL:        GHBaseURL,
	}

	err := loader.Load(context.Background(), cfg)
	if err != nil {
		return nil, err
	}

	return cfg, err
}
