package dist

import (
	"bytes"
	"context"
	"errors"
	"fmt"
	"io/ioutil"
	"regexp"
	"strings"

	"a.yandex-team.ru/security/hector/internal/config"
	"a.yandex-team.ru/security/hector/internal/core"
	"a.yandex-team.ru/security/hector/internal/dist/pkgparser"
	"a.yandex-team.ru/security/libs/go/retry"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/libs/go/yahttp"
)

type (
	Package struct {
		pkgparser.Package
		Arch    string
		Project string
		URL     string
	}

	Client struct {
		retrier retry.Retrier
		baseURL string
	}

	RepoWalkFn func(pkg Package) (next bool)
)

var (
	ErrStopWalk = errors.New("stop iterate")

	projectsRe  *regexp.Regexp
	projectArch = map[string]string{
		"all":   "/stable/all/Packages.gz",
		"amd64": "/stable/amd64/Packages.gz",
	}
)

func init() {
	projectsRe = regexp.MustCompile(`(?m)^<a href="([^"]+)"`)
}

func NewClient(baseURL string) *Client {
	return &Client{
		retrier: retry.New(retry.WithAttempts(config.RetryCount)),
		baseURL: baseURL,
	}
}

func (c Client) ListProjects() (repos []string, err error) {
	work := func(ctx context.Context) error {
		resp, err := core.Client.Get(c.baseURL)
		if err != nil {
			return fmt.Errorf("failed to list dist projects: %s", err.Error())
		}
		defer yahttp.GracefulClose(resp.Body)

		if resp.StatusCode != 200 {
			return fmt.Errorf("failed to list dist projects, status code: %s", resp.Status)
		}

		body, err := ioutil.ReadAll(resp.Body)
		if err != nil {
			return fmt.Errorf("failed to list dist projects: %s", err.Error())
		}

		matches := projectsRe.FindAllStringSubmatch(string(body), -1)
		repos = make([]string, len(matches))
		for i, project := range matches {
			repos[i] = strings.Trim(project[1], " /")
		}
		return nil
	}

	err = c.retrier.Try(context.Background(), work)
	return
}

func (c Client) WalkProjects(walkFn RepoWalkFn) error {
	projects, err := c.ListProjects()
	if err != nil {
		return err
	}

	for _, project := range projects {
		// TODO(buglloc): WTF?!
		if project == "verstka" {
			continue
		}

		if err = c.WalkProject(project, walkFn); err != nil {
			return err
		}
	}
	return nil
}

func (c Client) WalkProject(project string, walkFn RepoWalkFn) error {
	for arch := range projectArch {
		if err := c.WalkProjectArch(project, arch, walkFn); err != nil {
			return err
		}
	}
	return nil
}

func (c Client) WalkProjectArch(project string, arch string, walkFn RepoWalkFn) error {
	latestPkgs := make(map[string]Package)
	uri := fmt.Sprintf("%s/%s%s", c.baseURL, project, projectArch[arch])
	resp, err := core.Client.Get(uri)
	if err != nil {
		simplelog.Warn("failed to get pkg list", "uri", uri, "err", err.Error())
		return nil
	}
	defer yahttp.GracefulClose(resp.Body)

	if resp.StatusCode == 404 {
		// Skip not found pkgs
		return nil
	}

	if resp.StatusCode != 200 {
		simplelog.Warn("failed to get pkg list", "uri", uri, "status_code", resp.Status)
		return nil
	}

	data, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		simplelog.Warn("failed to read pkg list", "uri", uri, "err", err.Error())
		return nil
	}

	pkgs, err := pkgparser.ParsePkgListGz(bytes.NewReader(data))
	if err != nil {
		simplelog.Warn("failed to parse pkg list", "uri", uri, "err", err.Error())
		return nil
	}

	for _, pkg := range pkgs {
		latestPkgs[pkg.Name] = Package{
			Project: project,
			Arch:    arch,
			Package: pkg,
			URL:     c.baseURL + pkg.Filename,
		}
	}

	for _, pkg := range latestPkgs {
		if !walkFn(pkg) {
			return ErrStopWalk
		}
	}
	return nil
}

func (p Package) FullName() string {
	return p.Project + "/" + p.Arch + "/" + p.Name
}
