package pkgparser

import (
	"bufio"
	"compress/gzip"
	"errors"
	"io"
	"strconv"
	"strings"

	"a.yandex-team.ru/security/libs/go/simplelog"
)

type (
	Package struct {
		Name       string
		Version    string
		Maintainer string
		Filename   string
		Size       int
	}
)

var (
	filenameStr   = "Filename: "
	maintainerStr = "Maintainer: "
	versionStr    = "Version: "
	pkgStr        = "Package: "
	sizeStr       = "Size: "
)

func ParsePkgListGz(input io.Reader) (pkgs []Package, err error) {
	reader, rErr := gzip.NewReader(input)
	if rErr != nil {
		err = rErr
		return
	}

	return ParsePkgList(reader)
}

func ParsePkgList(input io.Reader) (pkgs []Package, err error) {
	scanner := bufio.NewScanner(input)
	var lines []string
	for scanner.Scan() {
		line := scanner.Text()
		if len(line) != 0 {
			lines = append(lines, line)
		} else {
			pkg, err := parsePkg(lines)
			lines = lines[:0]
			if err != nil {
				simplelog.Warn(err.Error())
				continue
			}

			pkgs = append(pkgs, pkg)
		}
	}

	//err = scanner.Err()
	return
}

func parsePkg(lines []string) (pkg Package, err error) {
	for _, line := range lines {
		switch {
		case strings.HasPrefix(line, filenameStr):
			pkg.Filename = line[len(filenameStr):]
		case strings.HasPrefix(line, maintainerStr):
			lIndex := strings.LastIndex(line, "<")
			rIndex := strings.LastIndex(line, ">")
			if lIndex == -1 || rIndex == -1 || lIndex > rIndex {
				pkg.Maintainer = line[len(maintainerStr):]
			} else {
				pkg.Maintainer = line[lIndex+1 : rIndex]
			}
		case strings.HasPrefix(line, versionStr):
			pkg.Version = line[len(versionStr):]
		case strings.HasPrefix(line, pkgStr):
			pkg.Name = line[len(pkgStr):]
		case strings.HasPrefix(line, sizeStr):
			if size, err := strconv.Atoi(line[len(sizeStr):]); err == nil {
				pkg.Size = size
			}
		}
	}

	if pkg.Filename == "" || pkg.Name == "" {
		//err = errors.New("failed to parse pkg: \n" + string(bytes.Join(lines, []byte{'\n'})))
		err = errors.New("failed to parse pkg")
	}
	return
}
