package gitlab

import (
	"crypto/tls"
	"fmt"
	"net/http"
	"net/url"
	"strings"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
)

type Client struct {
	httpc *resty.Client
}

func NewClient(authToken string, baseURL *url.URL) *Client {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		panic(fmt.Sprintf("failed to initialize cert pool: %v", err))
	}

	httpc := resty.New().
		SetTLSClientConfig(&tls.Config{RootCAs: certPool}).
		SetRetryCount(5).
		SetRedirectPolicy(resty.NoRedirectPolicy()).
		SetBaseURL(baseURL.String()).
		SetHeader("private-token", authToken).
		SetHeader("User-Agent", "Hector <security@yandex-team.ru>")

	return &Client{
		httpc: httpc,
	}
}

// Projects returns a map of repositories indexed by repository id.
func (c Client) Projects() (map[int]Project, error) {
	repositories := make(map[int]Project)
	uri := "/api/v4/projects"
	for uri != "" {
		var projects []Project
		rsp, err := c.httpc.R().SetResult(&projects).Get(uri)
		if err != nil {
			return nil, err
		}

		if !rsp.IsSuccess() {
			return nil, fmt.Errorf("non-200 status code: %s", rsp.Status())
		}

		for _, p := range projects {
			repositories[p.ID] = p
		}

		uri = nextPage(rsp.Header())
	}

	return repositories, nil
}

// GetProject returns a repository representation for the given Stash Project key and repository slug.
func (c Client) GetProject(name string) (Project, error) {
	var project Project
	rsp, err := c.httpc.R().
		SetResult(&project).
		SetPathParam("id", name).
		Get("/api/v4/projects/{id}")
	if err != nil {
		return project, err
	}

	if !rsp.IsSuccess() {
		return project, fmt.Errorf("non-200 status code: %s", rsp.Status())
	}

	return project, nil
}

// WalkProjects walk though stash repositories.
func (c Client) WalkProjects(walkFn RepoWalkFn) error {
	uri := "/api/v4/projects"
	for uri != "" {
		var projects []Project
		rsp, err := c.httpc.R().SetResult(&projects).Get(uri)
		if err != nil {
			return err
		}

		if !rsp.IsSuccess() {
			return fmt.Errorf("non-200 status code: %s", rsp.Status())
		}

		for _, p := range projects {
			if !walkFn(p) {
				return nil
			}
		}

		uri = nextPage(rsp.Header())
	}

	return nil
}

// HaveCommitsSince returns if repo have changes since date.
func (c Client) HaveCommitsSince(projectName string, ref string) (bool, error) {
	var commits []Commit
	rsp, err := c.httpc.R().
		SetResult(&commits).
		SetPathParam("id", projectName).
		Get("/api/v4/projects/{id}/repository/commits")
	if err != nil {
		return false, err
	}

	if !rsp.IsSuccess() {
		return false, fmt.Errorf("non-200 status code: %s", rsp.Status())
	}

	for _, commit := range commits {
		if commit.ID == ref {
			return true, nil
		}
	}

	return true, nil
}

// FullName construct repo name from the repository metadata.
func (repo Repository) FullName() string {
	return repo.Name
}

// IsFork returns true if repo is fork of another.
func (p Project) IsFork() bool {
	return p.ForkOf.ID != 0
}

func nextPage(header http.Header) string {
	for _, link := range strings.Split(header.Get("Link"), ",") {
		segments := strings.Split(strings.TrimSpace(link), ";")

		// link must at least have href and rel
		if len(segments) < 2 {
			continue
		}

		// ensure href is properly formatted
		if !strings.HasPrefix(segments[0], "<") || !strings.HasSuffix(segments[0], ">") {
			continue
		}

		// try to pull out page parameter
		u := segments[0][1 : len(segments[0])-1]
		if u == "" {
			continue
		}

		for _, segment := range segments[1:] {
			switch strings.TrimSpace(segment) {
			case `rel="next"`:
				return u
			}
		}
	}

	return ""
}
