package gitutils

import (
	"errors"
	"fmt"
	"os"
	"os/exec"
	"path"
	"regexp"
	"strings"
	"time"

	"github.com/karlseguin/ccache/v2"

	"a.yandex-team.ru/security/hector/internal/procutils"
	"a.yandex-team.ru/security/libs/go/semver"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

const (
	recommendedVersionConstruct = ">=2.3.0"
	blameTTL                    = time.Hour * 2
)

var (
	blameCache = ccache.New(ccache.Configure().MaxSize(1000))
)

func CheckVersion() error {
	stdout, ok := runGitCommand("/", "--version")
	if !ok {
		return nil
	}

	parseVersion := regexp.MustCompile(`git\s+version\s+((?:\d+)(?:\.\d+)?(?:\.\d+)?)`)
	versions := parseVersion.FindStringSubmatch(stdout)
	if len(versions) == 0 {
		return errors.New("failed to check git version (parsing error)")
	}

	rawVersion := versions[1]
	version, err := semver.NewVersion(rawVersion)
	if err != nil {
		return fmt.Errorf("failed to check git version %s (%s)", rawVersion, err.Error())
	}

	recommendedVersion, _ := semver.NewConstraint(recommendedVersionConstruct)
	if !recommendedVersion.Check(version) {
		return fmt.Errorf("git version '%s' are lower than recommended, please update git to '%s' or above",
			rawVersion, recommendedVersionConstruct)
	}

	return nil
}

func GetReference(repoPath string) string {
	stdout, ok := runGitCommand(repoPath, "rev-parse", "HEAD")
	if !ok {
		return ""
	}

	return strings.TrimSpace(stdout)
}

func LineAuthor(workDir, relativePath string, lineNo int) string {
	key := path.Join(workDir, relativePath)
	cachedBlame := blameCache.Get(key)
	var stdout string
	var blame []string
	if cachedBlame != nil && !cachedBlame.Expired() {
		blame = cachedBlame.Value().([]string)
	} else {
		var ok bool
		stdout, ok = runGitCommand(workDir, "blame", "-e", "-c", relativePath)
		if !ok {
			return ""
		}

		blame = parseBlame(stdout)
		blameCache.Set(key, blame, blameTTL)
	}

	if lineNo > len(blame) {
		simplelog.Error("blame line mismatch", "path", key, "line_no", lineNo, "blame_lines", len(blame))
		return ""
	}

	if lineNo > 0 {
		return blame[lineNo-1]
	}

	return blame[0]
}

func Checkout(cloneURL, target string, shallow bool) error {
	if _, err := os.Stat(target); err == nil {
		err := os.RemoveAll(target)
		if err != nil {
			return fmt.Errorf("failed to remove repo directory '%s': %s", target, err.Error())
		}
		simplelog.Info("Removed repo directory: " + target)
	}

	args := []string{"clone", "--quiet"}
	if shallow {
		args = append(args, "--depth=1")
		args = append(args, "--single-branch")
	}
	args = append(args, cloneURL)
	args = append(args, target)

	_, ok := runGitCommand("", args...)
	if !ok {
		return errors.New("failed to checkout")
	}

	return nil
}

func Unshallow(target string) error {
	_, ok := runGitCommand(target, "pull", "--unshallow", "--no-tags", "--quiet")
	if !ok {
		return errors.New("failed to unshallow repo")
	}

	return nil
}

func runGitCommand(workDir string, args ...string) (string, bool) {
	cmd := exec.Command("git", args...)
	if workDir != "" {
		cmd.Dir = workDir
	}
	cmd.Env = append(os.Environ(), "GIT_TERMINAL_PROMPT=0")
	res, err := procutils.RunCmd(cmd)
	if err != nil {
		simplelog.Error("git command failed", "command", args, "err", err.Error())
		return "", false
	}

	if res.ExitCode != 0 {
		simplelog.Error("git command failed", "command", args, "stderr", res.Stderr)
		return "", false
	}

	return res.Stdout, true
}

func parseBlame(blame string) []string {
	/*
	   7500447d        (<buglloc@yandex.ru>    2017-07-27 12:32:29 +0300       1)package main
	   7500447d        (<buglloc@yandex.ru>    2017-07-27 12:32:29 +0300       2)
	   7500447d        (<buglloc@yandex.ru>    2017-07-27 12:32:29 +0300       3)import "a.yandex-team.ru/security/hector/commands"
	   7500447d        (<buglloc@yandex.ru>    2017-07-27 12:32:29 +0300       4)
	   7500447d        (<buglloc@yandex.ru>    2017-07-27 12:32:29 +0300       5)func main() {
	   7500447d        (<buglloc@yandex.ru>    2017-07-27 12:32:29 +0300       6)      commands.Execute()
	   7500447d        (<buglloc@yandex.ru>    2017-07-27 12:32:29 +0300       7)}
	*/
	lines := strings.Split(blame, "\n")
	result := make([]string, len(lines))
	for i, l := range lines {
		raw := strings.SplitN(l, "\t", 3)
		if len(raw) > 1 {
			rawEmail := raw[1]
			result[i] = rawEmail[2 : len(rawEmail)-1]
		}
	}
	return result
}
