package bbrepo

import (
	"errors"
	"fmt"
	"net/url"
	"path"
	"strings"

	"a.yandex-team.ru/security/hector/internal/config"
	"a.yandex-team.ru/security/hector/internal/gitutils"
	"a.yandex-team.ru/security/hector/internal/remote"
	"a.yandex-team.ru/security/hector/internal/stash"
)

type BitbucketRepo struct {
	bbRepo      stash.Repository
	cloned      bool
	shallowRepo bool
	local       string
	name        string
	cloneURL    string
	htmlURL     string
	reference   string
}

func NewBitbucketRepo(bbRepo stash.Repository) *BitbucketRepo {
	return &BitbucketRepo{
		bbRepo: bbRepo,
	}
}

func (r *BitbucketRepo) Name() string {
	if r.name == "" {
		r.name = r.bbRepo.FullName()
	}
	return r.name
}

func (r *BitbucketRepo) Parent() string {
	return ""
}

func (r *BitbucketRepo) ProjectURL() string {
	if r.htmlURL == "" {
		r.htmlURL = r.bbRepo.HTMLURL()
		if r.htmlURL == "" {
			r.htmlURL = fmt.Sprintf("%s/projects/%s/repos/%s/browse",
				config.BBBaseURL, url.PathEscape(r.bbRepo.Project.Key), url.PathEscape(r.bbRepo.Name))
		}
	}
	return r.htmlURL
}

func (r *BitbucketRepo) CloneURL() string {
	if r.cloneURL == "" {
		if config.UseSSH {
			r.cloneURL = r.bbRepo.SSHURL()
		} else {
			r.cloneURL = r.bbRepo.HTTPURL()
		}
	}
	return r.cloneURL
}

func (r *BitbucketRepo) PathToURL(relativePath string, lineNo int) string {
	return fmt.Sprintf("%s/%s?until=%s#%d",
		r.ProjectURL(), relativePath, r.Reference(), lineNo)
}

func (r *BitbucketRepo) Checkout(target string, shallow bool) (skip bool, err error) {
	if err = gitutils.Checkout(r.CloneURL(), target, shallow); err != nil {
		return
	}

	r.shallowRepo = true
	r.cloned = true
	r.local = target
	r.reference = gitutils.GetReference(r.local)
	return
}

func (r *BitbucketRepo) Author(relativePath string, lineNo int) (author string, err error) {
	// Rewrite to Stash API?
	// https://docs.atlassian.com/DAC/rest/stash/3.11.6/stash-rest.html#idp3106640

	if !r.cloned {
		err = errors.New("can't get author on shallowed repo")
		return
	}

	if r.shallowRepo {
		if err = r.unshallow(); err != nil {
			return
		}
		r.shallowRepo = false
	}

	author = gitutils.LineAuthor(r.local, relativePath, lineNo)
	return
}

func (r *BitbucketRepo) Owners() (owners []string, err error) {
	//TODO(Buglloc): implement
	return
}

func (r *BitbucketRepo) Reference() string {
	if r.reference == "" {
		r.reference = gitutils.GetReference(r.local)
	}
	return r.reference
}

func (r *BitbucketRepo) unshallow() error {
	return gitutils.Unshallow(r.local)
}

func (r *BitbucketRepo) GenResultPath(resultDir string) string {
	baseName := strings.Replace(r.Name(), "/", ":", -1)
	return path.Join(resultDir, baseName+".json")
}

func (r *BitbucketRepo) GenLocalPath(workDir string) string {
	return path.Join(workDir, r.Name())
}

func (r *BitbucketRepo) IsPrivate() bool {
	return !r.bbRepo.Public
}

func (r *BitbucketRepo) Export() remote.ExportedRepo {
	owners, _ := r.Owners()

	return remote.ExportedRepo{
		Name:       r.name,
		Parent:     r.Parent(),
		CloneURL:   r.cloneURL,
		ProjectURL: r.ProjectURL(),
		Owners:     owners,
		Private:    r.IsPrivate(),
	}
}
